<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2026 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Component\Route66\Administrator\Helper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

use Firecoders\Component\Route66\Administrator\Optimizer\CSSOptimizer;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;

class PerformanceHelper
{
    public static function optimize(): void
    {
        $application = Factory::getApplication();

        if (!$application->isClient('site')) {
            return;
        }

        if ($application->input->getMethod() !== 'GET') {
            return;
        }

        $document = Factory::getDocument();

        if ($document->getType() !== 'html') {
            return;
        }

        $params = ComponentHelper::getParams('com_route66');

        if (!$params->get('iframe_facades') && !$params->get('iframes_lazy_load') && !$params->get('images_lazy_load') && !$params->get('optimize_css')) {
            return;
        }

        libxml_use_internal_errors(true);
        $dom                     = new \DOMDocument('1.0', 'UTF-8');
        $dom->preserveWhiteSpace = false;
        $dom->formatOutput       = false;
        $dom->loadHTML('<?xml encoding="UTF-8">' . $application->getBody(), LIBXML_SCHEMA_CREATE);

        $protected = self::prepareProtectedBlocks($dom);

        if ($params->get('iframe_facades')) {
            self::facadeIframes($dom);
        }

        if ($params->get('iframes_lazy_load')) {
            self::lazyLoadIframes($dom);
        }

        if ($params->get('images_lazy_load')) {
            self::lazyLoadImages($dom);
        }

        if ($params->get('optimize_css')) {
            self::optimizeStyles($dom);
        }

        // Restore protected blocks
        self::restoreProtectedBlocks($dom, $protected);

        $buffer = $dom->saveHTML();
        $buffer = str_replace('<?xml encoding="UTF-8">', '', $buffer);
        $buffer = html_entity_decode($buffer, ENT_HTML5, 'UTF-8');

        $application->setBody($buffer);
    }

    public static function assets(): void
    {
        $application = Factory::getApplication();

        if (!$application->isClient('site')) {
            return;
        }

        if ($application->input->getMethod() !== 'GET') {
            return;
        }

        $document = Factory::getDocument();

        if ($document->getType() !== 'html') {
            return;
        }

        $params = ComponentHelper::getParams('com_route66');

        if (!$params->get('iframe_facades')) {
            return;
        }

        $wa = $document->getWebAssetManager();
        $wa->registerAndUseScript('route66.lite-youtube', 'route66/lite-youtube/lite-youtube.min.js', [], ['type' => 'module']);
        $wa->registerAndUseScript('route66.lite-vimeo', 'route66/lite-vimeo/lite-vimeo.min.js', [], ['type' => 'module']);
    }


    protected static function lazyLoadImages(\DOMDocument $document): void
    {
        $params = ComponentHelper::getParams('com_route66');

        $mode      = $params->get('images_lazy_load_mode');
        $className = $params->get('images_lazy_load_classname');

        $images = $document->getElementsByTagName('img');

        for ($i = $images->length; --$i >= 0;) {

            $image = $images->item($i);

            if ($mode && $className) {

                $class   = $image->getAttribute('class');
                $classes = explode(' ', $class);
                $classes = array_filter($classes);

                if ($mode === 'inclusive' && !\in_array($className, $classes)) {
                    continue;
                } elseif ($mode === 'exclusive' && \in_array($className, $classes)) {
                    continue;
                }
            }

            $image->setAttribute('loading', 'lazy');
        }
    }

    protected static function lazyLoadIframes(\DOMDocument $document): void
    {
        $params = ComponentHelper::getParams('com_route66');

        $mode      = $params->get('iframes_lazy_load_mode');
        $className = $params->get('iframes_lazy_load_classname');

        $iframes = $document->getElementsByTagName('iframe');

        for ($i = $iframes->length; --$i >= 0;) {

            $iframe = $iframes->item($i);

            if ($mode && $className) {

                $class   = $iframe->getAttribute('class');
                $classes = explode(' ', $class);
                $classes = array_filter($classes);

                if ($mode === 'inclusive' && !\in_array($className, $classes)) {
                    continue;
                } elseif ($mode === 'exclusive' && \in_array($className, $classes)) {
                    continue;
                }
            }

            $iframe->setAttribute('loading', 'lazy');
        }
    }

    protected static function facadeIframes(\DOMDocument $document)
    {
        $params = ComponentHelper::getParams('com_route66');

        $mode      = $params->get('iframe_facades_mode');
        $className = $params->get('iframe_facades_classname');

        $iframes = $document->getElementsByTagName('iframe');

        for ($i = $iframes->length; --$i >= 0;) {

            $iframe = $iframes->item($i);

            if ($mode && $className) {

                $class   = $iframe->getAttribute('class');
                $classes = explode(' ', $class);
                $classes = array_filter($classes);

                if ($mode === 'inclusive' && !\in_array($className, $classes)) {
                    continue;
                } elseif ($mode === 'exclusive' && \in_array($className, $classes)) {
                    continue;
                }
            }

            $src = $iframe->getAttribute('src');

            if (str_contains($src, '//www.youtube.com/embed/') || str_contains($src, '//www.youtube-nocookie.com/embed/')) {
                self::setYoutubeFacade($document, $iframe);
            } elseif (str_contains($src, '//player.vimeo.com/video/')) {
                self::setVimeoFacade($document, $iframe);
            }
        }
    }

    protected static function setYoutubeFacade(\DOMDocument $document, \DomElement $iframe)
    {
        $src = $iframe->getAttribute('src');

        $parsed = parse_url($src);
        $host   = $parsed['host'];
        $path   = $parsed['path'];
        parse_str($parsed['query'], $query);

        $parts   = array_values(array_filter(explode('/', $path)));
        $videoId = array_pop($parts);

        if (!$videoId) {
            return;
        }

        if ($videoId === 'videoseries') {
            return;
        }

        $video = $document->createElement('lite-youtube');
        $video->setAttribute('videoid', $videoId);

        $playlistId = $query['list'] ?? '';
        if ($playlistId) {
            $video->setAttribute('playlistid', $playlistId);
        }

        $start = $query['start'] ?? 0;
        if ($start) {
            $video->setAttribute('videoStartAt', $start);
        }

        $title = $iframe->getAttribute('title');
        if ($title) {
            $video->setAttribute('videotitle', $title);
        }

        if ($host === 'www.youtube-nocookie.com') {
            $video->setAttribute('nocookie', true);
        }

        $iframe->parentNode->replaceChild($video, $iframe);
    }

    protected static function setVimeoFacade(\DOMDocument $document, \DomElement $iframe)
    {
        $src = $iframe->getAttribute('src');

        $parsed = parse_url($src);
        $path   = $parsed['path'];
        $parts  = array_values(array_filter(explode('/', $path)));

        $prefix = $parts[0] ?? '';

        if ($prefix !== 'video') {
            return;
        }

        $videoId = $parts[1] ?? '';

        if (!$videoId) {
            return;
        }

        $video = $document->createElement('lite-vimeo');
        $video->setAttribute('videoid', $videoId);

        $title = $iframe->getAttribute('title');
        if ($title) {
            $video->setAttribute('videotitle', $title);
        }

        $iframe->parentNode->replaceChild($video, $iframe);
    }

    protected static function optimizeStyles(\DOMDocument $document)
    {
        $xpath  = new \DOMXpath($document);
        $styles = $xpath->query('//link[@rel="stylesheet"] | //style');

        $optimizer = new CSSOptimizer();
        $processed = [];

        foreach ($styles as $key => $style) {
            $result = $optimizer->add($style);
            if ($result) {
                $processed[] = $key;
            }
        }

        $css = $optimizer->combine();

        if (!$css) {
            return;
        }

        foreach ($styles as $key => $style) {
            if (\in_array($key, $processed)) {
                $style->parentNode->removeChild($style);
            }
        }

        $head  = $document->getElementsByTagName('head')->item(0);
        $style = $document->createElement('style', $css);
        $head->appendChild($style);
    }

    /**
     * Replace <pre>, <code>, <textarea> with comment placeholders and return a mapping
     * from placeholder key to cloned DOMNode for restoration later.
     *
     * This is DOM-based (no regex), avoiding PCRE limits and round-trip issues.
     *
     * @return array<string, \DOMNode> placeholderKey => clonedNode
     */
    private static function prepareProtectedBlocks(\DOMDocument $dom): array
    {
        $protected = [];

        $xpath = new \DOMXPath($dom);

        // Collect targets (NodeList is live; convert to array first)
        $nodeList = $xpath->query('//pre | //code | //textarea');
        if (!$nodeList || $nodeList->length === 0) {
            return $protected;
        }

        $targets = [];
        foreach ($nodeList as $n) {
            $targets[] = $n;
        }

        // Only protect "outermost" matches: if a node is inside a target, skip it.
        $isInsideTarget = function (\DOMNode $node): bool {
            $p = $node->parentNode;
            while ($p instanceof \DOMElement) {
                $tag = strtolower($p->tagName);
                if ($tag === 'pre' || $tag === 'code' || $tag === 'textarea') {
                    return true;
                }
                $p = $p->parentNode;
            }
            return false;
        };

        $i = 0;

        foreach ($targets as $node) {
            // Skip nested code-like blocks (e.g. <code> inside <pre>)
            if ($isInsideTarget($node)) {
                continue;
            }

            if (!$node->parentNode) {
                continue;
            }

            $key         = "ROUTE66_BLOCK_$i";
            $placeholder = $dom->createComment($key);

            // Clone before replacing
            $clone = $node->cloneNode(true);

            $node->parentNode->replaceChild($placeholder, $node);

            $protected[$key] = $clone;
            $i++;
        }

        return $protected;
    }

    /**
     * Restore protected nodes by replacing placeholders with the stored clones.
     *
     * @param array<string, \DOMNode> $protected placeholderKey => clonedNode
     */
    private static function restoreProtectedBlocks(\DOMDocument $dom, array $protected): void
    {
        if (!$protected) {
            return;
        }

        $xpath = new \DOMXPath($dom);

        // Find our placeholders (comment nodes)
        $commentNodes = $xpath->query('//comment()[starts-with(normalize-space(.), "ROUTE66_BLOCK_")]');
        if (!$commentNodes || $commentNodes->length === 0) {
            return;
        }

        // Convert to array because we mutate the DOM
        $comments = [];
        foreach ($commentNodes as $c) {
            $comments[] = $c;
        }

        foreach ($comments as $comment) {
            $key = trim((string) ($comment->nodeValue ?? ''));

            if ($key === '' || !isset($protected[$key])) {
                continue;
            }

            if (!$comment->parentNode) {
                continue;
            }

            // Must clone; same node instance cannot be inserted multiple times
            $replacement = $protected[$key]->cloneNode(true);
            $comment->parentNode->replaceChild($replacement, $comment);
        }
    }
}
