<?php

namespace Spatie\Browsershot;

/**
 * Class with all outputs generated by puppeteer
 *
 * All present data is always relative to the last browser call.
 *
 * This object contains:
 *
 * - consoleMessages: messages generated with console calls
 * - requestsList: list of all requests made
 * - failedRequests: list of all failed requests
 * - result: result of the last operation called
 * - exception: string representation of the exception generated, if any
 * - pageErrors: list of all page errors generated during the current command
 * - redirectHistory: list of all redirect in the page
 */
class ChromiumResult
{
    protected string $result;
    protected string|null $exception;

    /**
     * @var null|array{type: string, message: string, location: array, stackTrace: string}
     */
    protected null|array $consoleMessages;

    /**
     * @var null|array{url: string}
     */
    protected null|array $requestsList;

    /**
     * @var null|array{status: int, url: string}
     */
    protected null|array $failedRequests;

    /**
     * @var null|array{name: string, message: string}
     */
    protected null|array $pageErrors;

    /**
     * @var null|array{url: string, status: int, statusText: string, headers: array}
     */
    protected null|array $redirectHistory;

    public function __construct(array|null $output)
    {
        $this->result = $output['result'] ?? '';
        $this->exception = $output['exception'] ?? null;
        $this->consoleMessages = $output['consoleMessages'] ?? null;
        $this->requestsList = $output['requestsList'] ?? null;
        $this->failedRequests = $output['failedRequests'] ?? null;
        $this->pageErrors = $output['pageErrors'] ?? null;
        $this->redirectHistory = $output['redirectHistory'] ?? null;
    }

    public function getResult(): string
    {
        return $this->result;
    }

    public function getException(): string|null
    {
        return $this->exception;
    }

    /**
     * @return null|array{type: string, message: string, location: array, stackTrace: string}
     */
    public function getConsoleMessages(): array|null
    {
        return $this->consoleMessages;
    }

    /**
     * @return null|array{url: string}
     */
    public function getRequestsList(): array|null
    {
        return $this->requestsList;
    }

    /**
     * @return null|array{status: int, url: string}
     */
    public function getFailedRequests(): array|null
    {
        return $this->failedRequests;
    }

    /**
     * @return null|array{name: string, message: string}
     */
    public function getPageErrors(): array|null
    {
        return $this->pageErrors;
    }

    /**
     * @return null|array{url: string, status: int, statusText: string, headers: array}
     */
    public function getredirectHistory(): array|null
    {
        return $this->redirectHistory;
    }
}
