<?php
/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */
defined('_JEXEC') or die;

class Route66Router
{
    protected $rules = array();
    protected $parseVars = array();

    public function __construct()
    {
        JPluginHelper::importPlugin('route66');
        $dispatcher = JEventDispatcher::getInstance();
        $results = $dispatcher->trigger('onRoute66AddRules');
        foreach ($results as $result) {
            foreach ($result as $rule) {
                $this->rules[] = $rule;
            }
        }
        usort($this->rules, array($this, 'sortRulesByLength'));
    }

    public function sortRulesByLength($a, $b)
    {
        $aLength = $a->getLength();
        $bLength = $b->getLength();
        if ($aLength == $bLength) {
            return 0;
        }

        return ($aLength > $bLength) ? -1 : 1;
    }

    public function preBuildRoute($router, $uri)
    {
        // Get query
        $query = $uri->getQuery(true);

        // Iterate over rules
        foreach ($this->rules as $rule) {

            // Initialize process flag
            $process = true;

            // Iterate over rule's conditions to see if we should process or not
            foreach ($rule->getRouteVariables('conditions') as $name => $value) {
                if (!isset($query[$name]) || $query[$name] != $value) {
                    $process = false;
                    break;
                }
            }

            // Process if the conditions are met
            if ($process) {

                // Check if this is a direct menu link using the required variables. TODO: Add an option for that?
                $direct = $this->isDirectMenuLink($query, $rule->getRouteVariables('menu'));

                // If not, process
                if (!$direct) {

                    // Unset the Itemid TODO: Do this always? Is there something else we need to check?
                    if (isset($query['Itemid'])) {
                        unset($query['Itemid']);
                    }

                    // The new route is the pattern with the tokens replaced with the corresponsing variables
                    $route = str_replace($rule->getTokens(), $rule->getTokensValues($query), $rule->getPattern());

                    // Unset all route variables
                    foreach ($rule->getRouteVariables() as $variable => $value) {
                        if (isset($query[$variable])) {
                            unset($query[$variable]);
                        }
                    }

                    // Update the query
                    $uri->setQuery($query);

                    // Update the path
                    $uri->setPath($uri->getPath().'/'.$route);

                    // Set a variable to know which component this route concerns. Used in the postBuildRoute to replace the component/option part of the URL
                    $router->setVar('route66', $rule->getOption());
                }
            }
        }
    }

    public function postBuildRoute($router, $uri)
    {
        // Check if URL has been generated by Route66
        if ($router->getVar('route66')) {

            // Compute the component name without the com_ prefix
            $component = substr($router->getVar('route66'), 4);

            // Get path
            $path = $uri->getPath();

            // Remove the component/option part
            $path = str_replace('/component/'.$component.'/', '', $path);

            // Remove the language that is appended by Joomla! router because we have already handled that
            $language = plgSystemRoute66::getLanguage();
            if ($language->sef) {
                $parts = explode('/', $path);
                $parts = array_filter($parts);
                array_pop($parts);
                $path = implode('/', $parts);
            }

            // Update the path
            $uri->setPath($path);

            // Reset the processed flag
            $vars = $router->getVars();
            unset($vars['route66']);
            $router->setVars($vars, false);
        }
    }

    public function preParseRoute($router, $uri)
    {
        // Get path
        $path = $uri->getPath();

        // Strip the extension if required
        if (strrpos($path, '.') !== false) {
            $path = substr($path, 0, strrpos($path, '.'));
        }

        // Proceed only if path is not empty ( = home page )
        if ($path) {

            // Get application
            $application = JFactory::getApplication();

            // Get menu
            $menu = $application->getMenu();

            // Check for menu items matching this path
            $items = $menu->getItems('route', $path);

            // Process only if it's not a menu item
            if (!count($items)) {

                // Count path length
                $length = substr_count($path, '/');

                // Iterate over rules
                foreach ($this->rules as $rule) {

                    // Quick check - Rule and path lengths should be equal
                    if ($rule->getLength() === $length) {

                        // Check if rule matches
                        $matches = array();
                        preg_match_all($rule->getRegex(), $path, $matches, PREG_SET_ORDER);
                        if (is_array($matches) && count($matches)) {

                            // Get token values from the matching expression
                            $values = array_combine($rule->getTokens(), array_slice($matches[0], 1));

                            // Init router vars
                            $vars = array();

                            // Add the language variable if it is required
                            $language = plgSystemRoute66::getLanguage();
                            if ($language->sef) {
                                $parts = explode('/', $path);
                                $vars['lang'] = $parts[0];
                            }

                            // Iterate over the rule variables
                            foreach ($rule->getRouteVariables() as $key => $var) {

                                // If it's empty it's not required
                                if (!$var) {
                                    continue;
                                }

                                // If it's a dynamic variable fetch it from the rule. Otherwise it's plain text
                                if ($var == '@') {
                                    $value = $rule->getQueryValue($key, $values);
                                    $var = $value ? $value : '@';
                                    $vars[$key] = $var;
                                } else {
                                    $vars[$key] = $var;
                                }
                            }

                            // If we have not found the dynamic vars move to the next rule
                            if (in_array('@', $vars)) {
                                continue;
                            }

                            // Ensure that tokens match the actual values
                            $canonical = $uri->getScheme().'://'.$uri->getHost().JRoute::_('index.php?'.http_build_query($vars), false);
                            if (rtrim($canonical, '/') != rtrim(JUri::current(), '/')) {
                                continue;
                            }

                            // Restore Itemid so the module assignments keep working
                            $vars['Itemid'] = $rule->getItemid($vars);

                            // Keep the parse vars so we can use them on the next stage
                            $this->parseVars = $vars;

                            // Return
                            return $vars;
                        }
                    }
                }
            }
        }
    }

    public function postParseRoute($router, $uri)
    {

        // Set again the vars we found to avoid conflicts with Itemid vars
        $router->setVars($this->parseVars);

        // Restore the active menu item. This will also restore the menu item params. TODO: Do we need this?
        if (isset($this->parseVars['Itemid'])) {
            JFactory::getApplication()->getMenu()->setActive($this->parseVars['Itemid']);
        }
    }

    private function isDirectMenuLink($query, $variables)
    {
        // If we don't have an Itemid this is not a direct menu link
        if (!isset($query['Itemid']) || !$query['Itemid']) {
            return false;
        }

        // Get application
        $application = JFactory::getApplication();

        // Get menu
        $menu = $application->getMenu();

        // Get menu item
        $item = $menu->getItem($query['Itemid']);

        // Init result
        $isDirect = true;

        // Iterate over variables to check
        foreach ($variables as $variable) {

            // If the query variable is not present in the link then this is not a direct link
            if (!isset($item->query[$variable])) {
                $isDirect = false;
                break;
            }

            // Set the comparison values
            $check = $item->query[$variable];
            $value = $query[$variable];

            // Check if the value is an array
            if (is_array($check)) {
                if (count($check) == 1) {
                    $check = $check[0];
                } else {
                    $isDirect = false;
                    break;
                }
            }

            // Normalize numeric variables
            if (is_numeric($check) || strpos($check, ':') || is_numeric($value) || strpos($value, ':')) {
                $check = (int) $check;
                $value = (int) $value;
            }

            // Check that values match
            if ($check != $value) {
                $isDirect = false;
                break;
            }
        }

        // Return
        return $isDirect;
    }
}
