<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Component\CommentBox\Site\Model;

use Firecoders\Component\CommentBox\Administrator\Helper\CommentHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\User\CurrentUserTrait;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class CommentsModel extends ListModel
{
    use CurrentUserTrait;

    public function __construct($config = [])
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = [
                'id', 'comment.id',
                'state', 'comment.state',
                'created', 'comment.created',
                'created_by', 'comment.created_by',
            ];
        }

        parent::__construct($config);
    }


    protected function populateState($ordering = 'comment.id', $direction = 'ASC')
    {
        $app   = Factory::getApplication();
        $input = $app->getInput();

        $value = $input->get('limit', $app->get('list_limit', 0), 'uint');
        if ($value > 50) {
            $value = 50;
        }
        $this->setState('list.limit', $value);

        $value = $input->get('limitstart', 0, 'uint');
        $this->setState('list.start', 0);

        $value = $input->getCmd('itemComponent');
        $this->setState('filter.item_component', $value);

        $value = $input->getCmd('itemView');
        $this->setState('filter.item_view', $value);

        $value = $input->getInt('itemId');
        $this->setState('filter.item_id', $value);

        $this->setState('filter.parent_id', 0);

        $this->setState('filter.state', 1);

        $user = Factory::getApplication()->getIdentity();

        if ($user->authorise('core.edit', 'com_commentbox') || $user->authorise('core.delete', 'com_commentbox') || $user->authorise('core.edit.state', 'com_commentbox')) {
            $this->setState('filter.state', null);
        } elseif ($user->authorise('core.edit.own', 'com_commentbox') || $user->authorise('core.trash.own', 'com_commentbox') || $user->authorise('core.delete.own', 'com_commentbox')) {
            $this->setState('filter.own', $user->id);
        }

        $value = $input->getInt('start');
        $this->setState('filter.start', $value);

        $value = $input->getInt('id');
        $this->setState('filter.id', $value);

        $params = $app->getParams();
        $this->setState('params', $params);
    }


    protected function getStoreId($id = '')
    {
        $id .= ':' . serialize($this->getState('filter.state'));
        $id .= ':' . serialize($this->getState('filter.item_component'));
        $id .= ':' . serialize($this->getState('filter.item_view'));
        $id .= ':' . serialize($this->getState('filter.item_id'));
        $id .= ':' . serialize($this->getState('filter.parent_id'));
        $id .= ':' . serialize($this->getState('filter.created_by'));
        $id .= ':' . serialize($this->getState('filter.own'));
        $id .= ':' . serialize($this->getState('filter.start'));
        $id .= ':' . serialize($this->getState('filter.id'));

        return parent::getStoreId($id);
    }

    protected function getListQuery()
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->select($this->getState('list.select', 'comment.id, comment.parent_id, comment.text, comment.state, comment.name, comment.email, comment.upvotes, comment.downvotes, comment.score, comment.created, comment.created_by, comment.modified'));
        $query->from('#__commentbox_comments AS comment');
        $query->join('RIGHT', '#__commentbox_pages AS page ON page.id = comment.page_id');

        if (!$this->getState('filter.item_component')) {
            throw new \Exception(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        $query->where('page.component = ' . $db->q($this->getState('filter.item_component')));

        if (!$this->getState('filter.item_view')) {
            throw new \Exception(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        $query->where('page.view = ' . $db->q($this->getState('filter.item_view')));

        if (!$this->getState('filter.item_id')) {
            throw new \Exception(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        $query->where('page.key = ' . $db->q($this->getState('filter.item_id')));

        if (\is_array($this->getState('filter.parent_id'))) {
            $query->where('comment.parent_id IN (' . implode(',', $this->getState('filter.parent_id')) . ')');
        } elseif (is_numeric($this->getState('filter.parent_id'))) {
            $query->where('comment.parent_id = ' . $db->q($this->getState('filter.parent_id')));
        }

        if ($this->getState('filter.own')) {
            $query->where('(comment.state = '.$db->q(1).' OR  comment.created_by = ' . $db->q($this->getState('filter.own')) . ')');
        } elseif (is_numeric($this->getState('filter.state'))) {
            $query->where('comment.state = ' . $db->q($this->getState('filter.state')));
        } elseif (\is_array($this->getState('filter.state'))) {
            $query->where('comment.state IN ('.implode(',', $this->getState('filter.state')).')');
        }

        if ($this->getState('filter.created_by')) {
            $query->where('comment.created_by = ' . $db->q($this->getState('filter.created_by')));
        }

        if ($this->getState('filter.id')) {
            $query->where('comment.id <= ' . $db->q($this->getState('filter.id')));
            $this->setState('list.limit', 0);

        } elseif (is_numeric($this->getState('filter.start'))) {
            $query->where('comment.id >= ' . $db->q($this->getState('filter.start') + 1));
        }

        $query->join('LEFT', '#__commentbox_reactions_counters AS reactions ON reactions.comment_id = comment.id');
        $query->select('reactions.like AS reactions_like, reactions.dislike AS reactions_dislike, reactions.love AS reactions_love, reactions.funny AS reactions_funny, reactions.wow AS reactions_wow, reactions.sad AS reactions_sad, reactions.angry AS reactions_angry');

        $query->order($this->getState('list.ordering', 'comment.id') . ' ' . $this->getState('list.direction', 'ASC'));

        return $query;
    }

    public function getTotal()
    {
        $this->setState('filter.id', 0);
        $this->setState('filter.start', 0);
        $this->setState('filter.parent_id', 0);

        return (int) $this->_getListCount($this->_getListQuery());
    }

    public function getTree()
    {
        $rows    = $this->getItems();

        $mapping = [];
        foreach ($rows as $row) {
            CommentHelper::prepare($row);
            $mapping[$row->id] = $row;
        }

        $parentIds = array_keys($mapping);

        // Fetch replies recursively and build the comment tree
        $this->getReplies($mapping, $parentIds);

        // Create the tree structure
        $comments = [];
        foreach ($rows as $row) {
            $comments[] = $row;
        }

        return $comments;
    }

    public function getItems()
    {
        $items       = parent::getItems();
        $application = Factory::getApplication();

        $model = $application->bootComponent('commentbox')->getMVCFactory()->createModel('Comment', 'Administrator', ['ignore_request' => true]);
        $model->getFlags($items);

        $user = $application->getIdentity();

        if ($application->isClient('site') && !$user->guest) {

            $params = ComponentHelper::getParams('com_commentbox');

            if ($params->get('votes')) {
                $model = Factory::getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Vote', 'Administrator', ['ignore_request' => true]);
                $model->getUserVotes($items, $user->id);
            }

            if ($params->get('reactions')) {

                $model = Factory::getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Reaction', 'Administrator', ['ignore_request' => true]);
                $model->getUserReactions($items, $user->id);
            }
        }

        return $items;
    }

    protected function getReplies($mapping, $parentIds)
    {

        if (!\count($parentIds)) {
            return;
        }

        $this->setState('list.limit', 1000);
        $this->setState('list.start', 0);
        $this->setState('filter.id', 0);
        $this->setState('filter.parent_id', $parentIds);

        $replies  = $this->getItems();
        $replyIds = [];
        foreach ($replies as $reply) {

            $replyIds[] = $reply->id;

            CommentHelper::prepare($reply);

            if (isset($mapping[$reply->parent_id])) {
                $reply->level                            = $mapping[$reply->parent_id]->level + 1;
                $mapping[$reply->parent_id]->replies[]   = $reply;
                $mapping[$reply->id]                     = $reply;
            }
        }

        $this->getReplies($mapping, $replyIds);
    }
}
