<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Component\CommentBox\Site\Controller;

use Firecoders\Component\CommentBox\Administrator\Helper\CaptchaHelper;
use Firecoders\Component\CommentBox\Administrator\Helper\HashHelper;
use Firecoders\Component\CommentBox\Administrator\Helper\UrlHelper;
use Joomla\CMS\Access\Exception\NotAllowed;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\ApiController;
use Joomla\CMS\MVC\Controller\Exception;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Tobscure\JsonApi\Exception\InvalidParameterException;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class CommentsController extends ApiController
{
    protected $contentType  = 'comments';
    protected $default_view = 'comments';

    public function getModel($name = 'Comment', $prefix = 'Administrator', $config = ['ignore_request' => true])
    {
        return parent::getModel($name, $prefix, $config);
    }

    public function displayItem($id = null)
    {
        if (!$id) {
            throw new Exception\ResourceNotFound(Text::_('JLIB_APPLICATION_ERROR_RECORD'), 404);
        }

        $view  = $this->getView('comment', 'json');
        $model = $this->getModel('Comment');
        $model->setState('comment.id', $id);
        $view->setModel($model, true);
        $view->display();
    }

    protected function preprocessSaveData(array $data): array
    {
        if (isset($data['item']) && \is_array($data['item']) && isset($data['item']['component']) && isset($data['item']['view']) && isset($data['item']['key'])) {

            $model = $this->getModel('Page');
            $page  = $model->getItemByResource($data['item']['component'], $data['item']['view'], $data['item']['key']);

            if ($page && $page->state != 1) {
                throw new NotAllowed(Text::_('JLIB_APPLICATION_ERROR_ACCESS_FORBIDDEN'), 403);
            }

            $pageData          = $page ? (array) $page : $data['item'];
            $pageData['title'] = $data['item']['title'];
            $pageData['link']  = Uri::root().$data['item']['link'];
            $pageData['state'] = 1;

            $form      = $model->getForm($pageData, false);
            $validData = $model->validate($form, $pageData);
            if ($validData === false) {
                throw new InvalidParameterException($model->getError(), 400);
            }
            $model->save($validData);
            $data['page_id'] = $model->getState('page.id');
        }

        if (isset($data['text']) && $data['text']) {
            $data['text'] = ComponentHelper::filterText($data['text']);
        }

        $data['state'] = $this->allowEditState() ? 1 : 0;

        return $data;
    }

    protected function save($recordKey = null)
    {
        $data  = $this->input->get('data', json_decode($this->input->json->getRaw(), true), 'array');

        if (!$this->checkCsrf($data)) {
            throw new NotAllowed(Text::_('JINVALID_TOKEN_NOTICE'), 403);
        }

        if (!isset($data['item']) || !\is_array($data['item'])) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        if (!isset($data['hash']) || !\is_string($data['hash'])) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        if (!HashHelper::checkHash($data['hash'], $data['item'])) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        if (CaptchaHelper::isEnabled() && !CaptchaHelper::verify($data)) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        $id = parent::save($recordKey);

        $params = ComponentHelper::getParams('com_commentbox');

        if ($id && $this->allowUpload() && $params->get('image_uploads', 1)) {
            $this->moveUploadedImages($id);
        }

        return $id;
    }

    public function delete($id = null)
    {
        $data  = $this->input->get('data', json_decode($this->input->json->getRaw(), true), 'array');

        if (!$this->checkCsrf($data)) {
            throw new NotAllowed(Text::_('JINVALID_TOKEN_NOTICE'), 403);
        }

        if ($id === null) {
            $id = $this->input->get('id', 0, 'int');
        }

        if (!$this->allowDelete($id)) {
            throw new NotAllowed('JLIB_APPLICATION_ERROR_DELETE_NOT_PERMITTED', 403);
        }

        $model = $this->getModel('Comment');
        $pks   = [$id];
        if (!$model->delete($pks)) {
            throw new \RuntimeException(Text::_('JLIB_APPLICATION_ERROR_DELETE'), 500);
        }

        $this->app->setHeader('status', 204);
    }

    public function publish()
    {
        $data  = $this->input->get('data', json_decode($this->input->json->getRaw(), true), 'array');

        if (!$this->checkCsrf($data)) {
            throw new NotAllowed(Text::_('JINVALID_TOKEN_NOTICE'), 403);
        }

        if (!isset($data['id']) || !is_numeric($data['id'])) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        if (!$this->allowEditState()) {
            throw new NotAllowed(Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), 403);
        }

        $cid   = [$data['id']];
        $model = $this->getModel();
        $model->publish($cid, 1);

        $this->displayItem($data['id']);
    }

    public function unpublish()
    {
        $data  = $this->input->get('data', json_decode($this->input->json->getRaw(), true), 'array');

        if (!$this->checkCsrf($data)) {
            throw new NotAllowed(Text::_('JINVALID_TOKEN_NOTICE'), 403);
        }

        if (!isset($data['id']) || !is_numeric($data['id'])) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        if (!$this->allowEditState()) {
            throw new NotAllowed(Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), 403);
        }

        $cid   = [$data['id']];
        $model = $this->getModel();
        $model->publish($cid, 0);

        $this->displayItem($data['id']);
    }

    public function spam()
    {
        $data  = $this->input->get('data', json_decode($this->input->json->getRaw(), true), 'array');

        if (!$this->checkCsrf($data)) {
            throw new NotAllowed(Text::_('JINVALID_TOKEN_NOTICE'), 403);
        }

        if (!isset($data['id']) || !is_numeric($data['id'])) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        if (!$this->allowEditState()) {
            throw new NotAllowed(Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), 403);
        }

        $cid   = [$data['id']];
        $model = $this->getModel();
        $model->publish($cid, -1);

        $this->displayItem($data['id']);
    }

    public function trash()
    {
        $data  = $this->input->get('data', json_decode($this->input->json->getRaw(), true), 'array');

        if (!$this->checkCsrf($data)) {
            throw new NotAllowed(Text::_('JINVALID_TOKEN_NOTICE'), 403);
        }

        if (!isset($data['id']) || !is_numeric($data['id'])) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        if (!$this->allowTrash($data)) {
            throw new NotAllowed(Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), 403);
        }

        $cid   = [$data['id']];
        $model = $this->getModel();
        $model->setState('action', 'frontend.trash');
        $model->publish($cid, -2);

        $this->displayItem($data['id']);
    }

    protected function allowAdd($data = [])
    {
        $user = $this->app->getIdentity();

        return $user->authorise('core.create', 'com_commentbox');
    }

    protected function allowEdit($data = [], $key = 'id')
    {
        $user = $this->app->getIdentity();

        if ($user->authorise('core.edit', 'com_commentbox')) {
            return true;
        }

        if ($user->authorise('core.edit.own', 'com_commentbox')) {
            $model   = $this->getModel('Comment');
            $table   = $model->getTable();
            $table->load($data[$key]);

            return $table->created_by === $user->id;
        }

        return false;
    }

    protected function allowEditState($data = [], $key = 'id')
    {
        $user = $this->app->getIdentity();

        return $user->authorise('core.edit.state', 'com_commentbox');
    }

    protected function allowTrash($data = [], $key = 'id')
    {
        $user = $this->app->getIdentity();

        if ($user->authorise('core.edit.state', 'com_commentbox')) {
            return true;
        }

        if ($user->authorise('core.trash.own', 'com_commentbox')) {
            $model   = $this->getModel('Comment');
            $table   = $model->getTable();
            $table->load($data[$key]);

            return $table->created_by === $user->id;
        }

        return false;
    }

    protected function allowDelete($id)
    {
        $user = $this->app->getIdentity();

        if ($user->authorise('core.delete', 'com_commentbox')) {
            return true;
        }

        if ($user->authorise('core.delete.own', 'com_commentbox')) {
            $model   = $this->getModel('Comment');
            $table   = $model->getTable();
            $table->load($id);

            return $table->created_by === $user->id;
        }

        return false;
    }

    protected function allowUpload()
    {
        $user = $this->app->getIdentity();

        return $user->authorise('core.upload', 'com_commentbox');
    }


    protected function checkCsrf($data)
    {
        $token = Session::getFormToken();

        return isset($data[$token]) && $data[$token] === 1;
    }

    public function parseUrl()
    {
        $data  = $this->input->get('data', json_decode($this->input->json->getRaw(), true), 'array');

        if (!isset($data['uri']) || !isset($data['hash'])) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        if (!$data['uri'] || !$data['hash']) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        if (!HashHelper::checkHash($data['hash'], ['uri' => $data['uri']])) {
            throw new NotAllowed(Text::_('JLIB_APPLICATION_ERROR_ACCESS_FORBIDDEN'), 403);
        }

        [$component, $view, $key] = UrlHelper::parse($data['uri']);

        $response = [
            'component' => $component,
            'view'      => $view,
            'key'       => $key,
        ];

        echo json_encode($response, JSON_UNESCAPED_UNICODE);

        return $this;
    }

    private function moveUploadedImages($id)
    {
        $model   = $this->getModel('Comment');
        $comment = $model->getItem($id);

        if (!$comment) {
            return;
        }

        if (!$comment->id) {
            return;
        }

        if (!$comment->text) {
            return;
        }

        $doc = new \DOMDocument();
        $doc->loadHTML($comment->text);

        $searches     = [];
        $replacements = [];

        $folder = Factory::getDate($comment->created)->format('Y/m/d');
        $images = $doc->getElementsByTagName('img');

        foreach ($images as $image) {

            $src = (string) $image->getAttribute('src');

            if (!$src) {
                continue;
            }

            if (strpos(strtolower($src), 'http:') === 0 || strpos(strtolower($src), 'https:') === 0) {
                continue;
            }

            if (strpos($src, 'media/commentbox/tmp') === false) {
                continue;
            }

            $filename = basename(strtolower($src));
            $filename = File::makeSafe($filename);

            if (!$filename) {
                continue;
            }

            if (!File::exists(JPATH_SITE.'/media/commentbox/tmp/'.$filename)) {
                continue;
            }

            if (!Folder::exists(JPATH_SITE.'/media/commentbox/uploads/'.$folder.'/'.$comment->id)) {
                Folder::create(JPATH_SITE.'/media/commentbox/uploads/'.$folder.'/'.$comment->id);
            }

            if (File::move(JPATH_SITE.'/media/commentbox/tmp/'.$filename, JPATH_SITE.'/media/commentbox/uploads/'.$folder.'/'.$comment->id.'/'.$filename)) {
                $searches[]     = $src;
                $replacements[] = '/media/commentbox/uploads/'.$folder.'/'.$comment->id.'/'.$filename;
            }
        }

        if (\count($searches)) {
            $comment->text = str_replace($searches, $replacements, $comment->text);
            $table         = $model->getTable();
            $table->id     = $comment->id;
            $table->text   = $comment->text;
            $table->store();
        }

    }


    public function counters()
    {
        $data  = $this->input->get('data', json_decode($this->input->json->getRaw(), true), 'array');

        if (!$this->checkCsrf($data)) {
            throw new NotAllowed(Text::_('JINVALID_TOKEN_NOTICE'), 403);
        }

        if (!isset($data['urls']) || !\is_array($data['urls'])) {
            throw new InvalidParameterException(Text::_('COM_COMMENTBOX_ERROR_BAD_REQUEST'), 400);
        }

        $counters = [];

        foreach ($data['urls'] as $url) {

            [$component, $view, $key] = UrlHelper::parse($url);

            $model = $this->getModel('Page', 'Administrator');
            $page  = $model->getItemByResource($component, $view, $key);

            $counters[] = (object) [
                'url'   => $url,
                'count' => $page ? $page->comments_published : 0,
            ];
        }

        echo json_encode($counters, JSON_UNESCAPED_UNICODE);

        return $this;
    }

    public function gifs()
    {
        $data = $this->input->get('data', json_decode($this->input->json->getRaw(), true), 'array');

        if (!$this->checkCsrf($data)) {
            throw new NotAllowed(Text::_('JINVALID_TOKEN_NOTICE'), 403);
        }

        $search   = isset($data['search']) && \is_string($data['search']) ? $data['search'] : '';
        $language = str_replace('-', '_', $this->app->getLanguage()->getTag());
        $limit    = 60;
        $next     = isset($data['next']) && \is_string($data['next']) ? $data['next'] : '';

        $cache = Factory::getCache('com_commentbox.gifs', 'callback');
        $cache->setCaching(true);
        $cache->setLifeTime(60 * 24);
        $results = $cache->get(['\Firecoders\Component\CommentBox\Administrator\Helper\GifHelper', 'fetch'], [$search, $language, $limit, $next]);

        echo json_encode($results, JSON_UNESCAPED_UNICODE);

        return $this;
    }
}
