<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Component\CommentBox\Administrator\Table;

use Joomla\CMS\Factory;
use Joomla\CMS\Table\Table;
use Joomla\Database\DatabaseDriver;
use Joomla\Event\DispatcherInterface;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class CommentTable extends Table
{
    public function __construct(DatabaseDriver $db, DispatcherInterface $dispatcher = null)
    {
        parent::__construct('#__commentbox_comments', 'id', $db, $dispatcher);

        $this->setColumnAlias('published', 'state');

        $this->created = Factory::getDate()->toSql();
    }


    public function store($updateNulls = false)
    {
        $isNew = !$this->hasPrimaryKey();

        try {
            $this->_db->transactionStart();

            parent::store($updateNulls);

            if ($isNew) {

                $query = $this->_db->getQuery(true);
                $query->update($this->_db->qn('#__commentbox_pages'));
                $query->set($this->_db->qn('comments_total').' = '.$this->_db->qn('comments_total').' + 1');

                if ($this->state == 1) {
                    $query->set($this->_db->qn('comments_published').' = '.$this->_db->qn('comments_published').' + 1');
                } elseif ($this->state == 0) {
                    $query->set($this->_db->qn('comments_unpublished').' = '.$this->_db->qn('comments_unpublished').' + 1');
                } elseif ($this->state == -1) {
                    $query->set($this->_db->qn('comments_spam').' = '.$this->_db->qn('comments_spam').' + 1');
                } elseif ($this->state == -2) {
                    $query->set($this->_db->qn('comments_trashed').' = '.$this->_db->qn('comments_trashed').' + 1');
                }

                $query->where($this->_db->qn('id').' = '.$this->_db->q($this->page_id));
                $this->_db->setQuery($query);
                $this->_db->execute();

                $record = (object)['comment_id' => $this->id];
                $this->_db->insertObject('#__commentbox_reactions_counters', $record);
            }

            $this->_db->transactionCommit();

        } catch (Exception $e) {

            $this->_db->transactionRollback();
            return false;
        }

        return true;
    }

    public function publish($pks = null, $state = 1, $userId = 0)
    {
        $counters    = [];
        $commentIds  = [];
        $states      = [
            1  => 'published',
            0  => 'unpublished',
            -1 => 'spam',
            -2 => 'trashed',
        ];

        $query = $this->_db->getQuery(true);
        $query->select($this->_db->qn('id'));
        $query->select($this->_db->qn('state'));
        $query->select($this->_db->qn('page_id'));
        $query->from($this->_db->qn('#__commentbox_comments'));
        $query->where($this->_db->qn('state').'!='.$this->_db->q($state));
        $query->where($this->_db->qn('id').' IN ('.implode(',', $pks).')');

        $this->_db->setQuery($query);
        $rows = $this->_db->loadObjectList();

        foreach ($rows as $row) {

            $commentIds[] = $row->id;

            if (!isset($counters[$row->page_id])) {
                $counters[$row->page_id] = (object) [
                    'published'   => 0,
                    'unpublished' => 0,
                    'spam'        => 0,
                    'trashed'     => 0,
                ];
            }

            $key = $states[$state];
            $counters[$row->page_id]->$key++;
            $key = $states[$row->state];
            $counters[$row->page_id]->$key--;
        }

        $pageIds  = array_keys($counters);

        try {

            $this->_db->transactionStart();

            $query = $this->_db->getQuery(true);
            $query->update($this->_db->qn('#__commentbox_comments'));
            $query->set($this->_db->qn('state').' = '.$this->_db->q($state));
            $query->where($this->_db->qn('id').' IN ('.implode(',', $commentIds).')');
            $this->_db->setQuery($query);
            $this->_db->execute();

            $operator = $state == 1 ? '+' : '-';

            foreach ($pageIds as $pageId) {

                $pageCounters = $counters[$pageId];

                $query = $this->_db->getQuery(true);
                $query->update($this->_db->qn('#__commentbox_pages'));

                foreach ($pageCounters as $state => $value) {
                    if ($value == 0) {
                        continue;
                    }
                    $operator = $value > 0 ? '+' : '-';
                    $query->set($this->_db->qn('comments_'.$state).' = '.$this->_db->qn('comments_'.$state).' '.$operator.' '.abs($value));
                }

                $query->where($this->_db->qn('id').' = '.$this->_db->q($pageId));
                $this->_db->setQuery($query);
                $this->_db->execute();
            }

            $this->_db->transactionCommit();

        } catch (Exception $e) {

            $this->_db->transactionRollback();
            return false;
        }

        return true;
    }

    public function delete($pk = null)
    {
        $counters    = [];
        $states      = [
            1  => 'published',
            0  => 'unpublished',
            -1 => 'spam',
            -2 => 'trashed',
        ];

        $query = $this->_db->getQuery(true);
        $query->select($this->_db->qn('id'));
        $query->select($this->_db->qn('state'));
        $query->select($this->_db->qn('page_id'));
        $query->from($this->_db->qn('#__commentbox_comments'));
        if (\is_array($pk)) {
            $query->where($this->_db->qn('id').' IN ('.implode(',', $pks).')');
        } else {
            $query->where($this->_db->qn('id').' = '.$this->_db->q($pk));
        }
        $this->_db->setQuery($query);
        $rows = $this->_db->loadObjectList();

        foreach ($rows as $row) {

            if (!isset($counters[$row->page_id])) {
                $counters[$row->page_id] = (object) [
                    'published'   => 0,
                    'unpublished' => 0,
                    'spam'        => 0,
                    'trashed'     => 0,
                ];
            }

            $key = $states[$row->state];
            $counters[$row->page_id]->$key++;
        }

        $pageIds  = array_keys($counters);

        try {
            $this->_db->transactionStart();

            parent::delete($pk);

            foreach ($pageIds as $pageId) {

                $query = $this->_db->getQuery(true);
                $query->update($this->_db->qn('#__commentbox_pages'));

                $removed = 0;

                foreach ($counters[$pageId] as $state => $value) {

                    if ($value == 0) {
                        continue;
                    }

                    $query->set($this->_db->qn('comments_'.$state).' = '.$this->_db->qn('comments_'.$state).' - '.$value);

                    $removed = $removed + $value;
                }

                if ($removed) {
                    $query->set($this->_db->qn('comments_total').' = '.$this->_db->qn('comments_total').' - '.$removed);
                }

                $query->where($this->_db->qn('id').' = '.$this->_db->q($pageId));
                $this->_db->setQuery($query);
                $this->_db->execute();
            }

            $query = $this->_db->getQuery(true);
            $query->delete($this->_db->qn('#__commentbox_reactions_counters'));
            $query->where($this->_db->qn('comment_id').' = '.$this->_db->q($pk));
            $this->_db->setQuery($query);
            $this->_db->execute();

            $this->_db->transactionCommit();

        } catch (Exception $e) {

            $this->_db->transactionRollback();
            return false;
        }

        return true;
    }
}
