<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Component\CommentBox\Administrator\Model;

use Firecoders\Component\CommentBox\Administrator\Helper\UserHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\Uri\Uri;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class CommentsModel extends ListModel
{
    public function __construct($config = [])
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = [
                'id', 'comment.id',
                'text', 'comment.text',
                'state', 'comment.state',
                'page_id', 'comment.page_id',
                'flag_id', 'comment.flag_id',
                'parent_id', 'comment.parent_id',
                'flagged', 'comment.flagged',
                'created', 'comment.created',
                'created_by', 'comment.created_by',
                'reactions', 'comment.reactions',
                'votes', 'comment.votes',
                'score', 'comment.score',
                'contains',
            ];
        }

        parent::__construct($config);
    }

    protected function getListQuery()
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->select($this->getState('list.select', 'comment.*'));
        $query->from('#__commentbox_comments AS comment');

        if (\is_array($this->getState('filter.page_id'))) {
            $query->where('comment.page_id IN (' . implode(',', $this->getState('filter.page_id')) . ')');
        } elseif (is_numeric($this->getState('filter.page_id'))) {
            $query->where('comment.page_id = ' . $db->q($this->getState('filter.page_id')));
        }

        if (\is_array($this->getState('filter.parent_id'))) {
            $query->where('comment.parent_id IN (' . implode(',', $this->getState('filter.parent_id')) . ')');
        } elseif (is_numeric($this->getState('filter.parent_id'))) {
            $query->where('comment.parent_id = ' . $db->q($this->getState('filter.parent_id')));
        }

        if (\is_array($this->getState('filter.flag_id')) && \count($this->getState('filter.flag_id'))) {
            $query->join('INNER', '#__commentbox_comments_flags_xref AS xref ON xref.comment_id = comment.id');
            $query->where('xref.flag_id IN (' . implode(',', $this->getState('filter.flag_id')) . ')');
            $query->group('comment.id');
        }

        if ($this->getState('filter.own')) {
            $query->where('(comment.state = 1 OR comment.created_by = ' . $db->q($this->getState('filter.own')) . ')');
        } elseif (is_numeric($this->getState('filter.state'))) {
            $query->where('comment.state = ' . $db->q($this->getState('filter.state')));
        }

        if ($this->getState('filter.created_by')) {
            $query->where('comment.created_by = ' . $db->q($this->getState('filter.created_by')));
        }

        if (\is_array($this->getState('filter.id'))) {
            $query->where('comment.id IN (' . implode(',', $this->getState('filter.id')) . ')');
        } elseif (is_numeric($this->getState('filter.id'))) {
            $query->where('comment.id = ' . $db->q($this->getState('filter.id')));
        }

        if ($this->getState('filter.created')) {
            $date = Factory::getDate();
            $date->sub(\DateInterval::createFromDateString($this->getState('filter.created')));
            $query->where('comment.created >= ' . $db->q($date->toSql()));
        }

        if (is_numeric($this->getState('filter.flagged'))) {
            $query->where('comment.flagged = ' . $db->q($this->getState('filter.flagged')));
        }

        $search = $this->getState('filter.search');

        if ($search) {
            if (is_numeric($search)) {
                $query->where($db->qn('comment.id') . ' = '.$db->q($search));
            } else {
                $search = '%' . trim($search) . '%';
                $query->where('(' .
                    $db->qn('comment.text') . ' LIKE '.$db->q($search) . ' OR ' .
                    $db->qn('comment.name') . ' LIKE '.$db->q($search) . ' OR ' .
                    $db->qn('comment.email') . ' LIKE '.$db->q($search) . ' OR ' .
                    $db->qn('comment.ip') . ' LIKE '.$db->q($search) . ' OR ' .
                    $db->qn('comment.user_agent') . ' LIKE '.$db->q($search) . ' OR ' .
                    $db->qn('user.name') . ' LIKE '.$db->q($search).' OR ' .
                    $db->qn('user.username') . ' LIKE '.$db->q($search).' OR ' .
                    $db->qn('user.email') . ' LIKE '.$db->q($search)
                .')');
            }
        }

        if ($contains = $this->getState('filter.contains')) {
            $condition = null;
            if ($contains === 'link') {
                $condition = '%' . trim('<a target="_blank" ') . '%';
            } elseif ($contains === 'mention') {
                $condition = '%' . trim('class="commentbox-mention"') . '%';
            } elseif ($contains === 'emoji') {
                $condition = '%' . trim('class="commentbox-emoji"') . '%';
            }
            if ($condition) {
                $query->where($db->qn('comment.text') . ' LIKE '.$db->q($condition));
            }
        }

        $query->join('LEFT', '#__commentbox_pages AS page ON page.id = comment.page_id');
        $query->select('page.title AS pageTitle');
        $query->select('page.link AS pageLink');

        $query->join('LEFT', '#__users AS user ON user.id = comment.created_by');
        $query->select('user.name AS userName');
        $query->select('user.email AS userEmail');

        $query->order($this->getState('list.ordering', 'comment.created') . ' ' . $this->getState('list.direction', 'DESC'));

        return $query;
    }


    protected function getStoreId($id = '')
    {
        $id .= ':' . $this->getState('filter.state');
        $id .= ':' . serialize($this->getState('filter.page_id'));
        $id .= ':' . serialize($this->getState('filter.id'));
        $id .= ':' . $this->getState('filter.search');
        $id .= ':' . serialize($this->getState('filter.parent_id'));
        $id .= ':' . serialize($this->getState('filter.flag_id'));
        $id .= ':' . $this->getState('filter.created_by');
        $id .= ':' . $this->getState('filter.created');
        $id .= ':' . $this->getState('filter.flagged');
        $id .= ':' . $this->getState('filter.contains');

        return parent::getStoreId($id);
    }


    public function getItems()
    {
        $items = parent::getItems();

        $params = ComponentHelper::getParams('com_commentbox');

        if ($params->get('ai_moderation', 0)) {
            $model  = Factory::getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Comment', 'Administrator', ['ignore_request' => true]);
            $model->getFlags($items);
        }

        foreach ($items as $item) {
            if (!$item->created_by) {
                $item->userName  = $item->name;
                $item->userEmail = $item->email;
            }

            $item->userAvatar = UserHelper::getAvatar($item->userEmail);
            $item->pageLink   = Uri::root().$item->pageLink;
            $item->text       = str_replace('src="/media/commentbox/uploads', 'src="'.Uri::root(true).'/media/commentbox/uploads', $item->text);
        }

        return $items;
    }


    public function getTable($type = 'Comment', $prefix = 'Administrator', $config = [])
    {
        return parent::getTable($type, $prefix, $config);
    }


    protected function populateState($ordering = 'comment.id', $direction = 'desc')
    {
        // Load the parameters.
        $this->setState('params', ComponentHelper::getParams('com_commentbox'));

        // Load the filter state.
        $search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
        $this->setState('filter.search', $search);

        $state = $this->getUserStateFromRequest($this->context . '.filter.state', 'filter_state', '', 'int');
        $this->setState('filter.state', $state);

        $createdBy = $this->getUserStateFromRequest($this->context . '.filter.created_by', 'filter_created_by', '', 'integer');
        $this->setState('filter.created_by', $createdBy);

        $pageId = $this->getUserStateFromRequest($this->context . '.filter.page_id', 'filter_page_id', '', 'integer');
        $this->setState('filter.page_id', $pageId);

        $contains = $this->getUserStateFromRequest($this->context . '.filter.contains', 'filter_contains');
        $this->setState('filter.contains', $contains);

        $flagged = $this->getUserStateFromRequest($this->context . '.filter.flagged', 'filter_flagged', '', 'int');
        $this->setState('filter.flagged', $flagged);

        $this->getUserStateFromRequest($this->context . '.filter.flag_id', 'filter_flag_id', []);

        // List state information.
        parent::populateState($ordering, $direction);
    }

    public function deletePageComments($pageIds)
    {
        $db = $this->getDatabase();

        try {
            $db->transactionStart();

            // Delete comments
            $query = $db->getQuery(true);
            $query->delete($db->qn('#__commentbox_comments'));
            $query->where($db->qn('page_id').' IN ('.implode(',', $pageIds).')');
            $db->setQuery($query);
            $db->execute();

            // Reset comment counters
            $query = $db->getQuery(true);
            $query->update($db->qn('#__commentbox_pages'));
            $query->set($db->qn('comments_total').' = 0');
            $query->set($db->qn('comments_published').' = 0');
            $query->set($db->qn('comments_unpublished').' = 0');
            $query->set($db->qn('comments_spam').' = 0');
            $query->set($db->qn('comments_trashed').' = 0');
            $query->where($db->qn('id').' IN ('.implode(',', $pageIds).')');

            $db->setQuery($query);
            $db->execute();

            $db->transactionCommit();

        } catch (Exception $e) {

            $db->transactionRollback();
            return false;
        }

        return true;
    }

    public function getFilterForm($data = [], $loadData = true)
    {
        $form = parent::getFilterForm($data, $loadData);

        $params = ComponentHelper::getParams('com_commentbox');

        if ($form && !$params->get('ai_moderation', 0)) {
            $form->removeField('flag_id', 'filter');
        }

        return $form;
    }
}
