<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Component\CommentBox\Administrator\Model;

use Firecoders\Component\CommentBox\Administrator\Helper\AntiSpamHelper;
use Firecoders\Component\CommentBox\Administrator\Helper\ModerationHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\Filesystem\Folder;
use Joomla\Utilities\IpHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class CommentModel extends AdminModel
{
    protected $text_prefix = 'COM_COMMENTBOX_COMMENT';

    public function getForm($data = [], $loadData = true)
    {
        $form = $this->loadForm('com_commentbox.comment', JPATH_ADMINISTRATOR.'/components/com_commentbox/forms/comment.xml', ['control' => 'jform', 'load_data' => $loadData]);

        if (empty($form)) {
            return false;
        }

        return $form;
    }

    public function save($data)
    {
        $application = Factory::getApplication();
        $user        = Factory::getApplication()->getIdentity();
        $date        = Factory::getDate()->toSql();

        $data['itemComponent'] = $application->input->getCmd('item_component');
        $data['itemView']      = $application->input->getCmd('item_view');
        $data['itemId']        = $application->input->getCmd('item_id');
        $data['ip']            = IpHelper::getIp();
        $data['user_agent']    = $application->input->server->getString('HTTP_USER_AGENT', '');

        if (isset($data['id']) && $data['id']) {
            $data['modified_by'] = $user->id;
            $data['modified']    = $date;
        } else {
            $data['created']    = $date;
            $data['created_by'] = $user->id;
            $data['name']       = $user->guest ? $data['name'] : $user->name;
            $data['email']      = $user->guest ? $data['email'] : $user->email;
        }

        if (AntiSpamHelper::isEnabled() && AntiSpamHelper::checkSpam($data)) {
            $data['state'] = -1;
        }

        $flagged = null;

        if (ModerationHelper::isEnabled($data['state'])) {

            $flagged = ModerationHelper::moderate($data);

            if (\is_bool($flagged)) {

                $data['flagged'] = $flagged ? 1 : 0;

                if (ModerationHelper::autoPublish() && $flagged === false) {
                    $data['state'] = 1;
                } elseif ($flagged === true) {
                    if (ModerationHelper::unpublish()) {
                        $data['state'] = 0;
                    } elseif (ModerationHelper::trash()) {
                        $data['state'] = -2;
                    } elseif (ModerationHelper::reject()) {
                        throw new \Exception(Text::_('COM_COMMENTBOX_COMMENT_REJECTED'), 422);
                    }
                }
            }
        }

        $result = parent::save($data);

        if ($result) {

            if ($flagged === null) {
                return $result;
            }

            $commentId = $this->getState('comment.id');

            if (!$commentId) {
                return $result;
            }

            $this->removeFlags($commentId);
            $this->addFlags($commentId, ModerationHelper::$flags);
        }

        return $result;
    }

    protected function loadFormData()
    {
        $app  = Factory::getApplication();
        $data = $app->getUserState('com_commentbox.edit.comment.data', []);

        if (empty($data)) {
            $data = $this->getItem();
        }

        $this->preprocessData('com_commentbox.comment', $data);

        return $data;
    }

    public function delete(&$pks)
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->select($db->qn('id'));
        $query->select($db->qn('created'));
        $query->from($db->qn('#__commentbox_comments'));
        $query->where($db->qn('id').' IN ('.implode(',', $pks).')');
        $db->setQuery($query);
        $comments = $db->loadObjectList();

        $mapping = [];

        foreach ($comments as $comment) {
            $mapping[$comment->id] = $comment->created;
        }

        $result = parent::delete($pks);

        if ($result) {

            foreach ($pks as $pk) {

                $this->removeFlags($pk);

                if (!isset($mapping[$pk])) {
                    continue;
                }

                $folder = Factory::getDate($mapping[$pk])->format('Y/m/d');

                if (Folder::exists(JPATH_SITE. '/media/commentbox/uploads/'.$folder.'/'.$pk)) {
                    Folder::delete(JPATH_SITE. '/media/commentbox/uploads/'.$folder.'/'.$pk);
                }
            }

        }

        return $result;
    }

    protected function canEditState($comment)
    {
        $user = $this->getCurrentUser();

        // Allow users in frontend to trash their own comments
        if ($this->getState('action', 'frontend.trash')) {
            return $user->authorise('core.edit.state', 'com_commentbox') || ($user->authorise('core.trash.own', 'com_commentbox') && $comment->created_by == $user->id);
        }

        return $user->authorise('core.edit.state', 'com_commentbox');
    }

    public function getFlags(&$comments)
    {
        $db = $this->getDatabase();

        $commentIds = array_map(function ($comment) {
            return $comment->id;
        }, $comments);

        if (empty($commentIds)) {
            return;
        }

        $query = $db->getQuery(true);
        $query->select($db->qn('flag.id'));
        $query->select($db->qn('flag.key'));
        $query->select($db->qn('xref.score'));
        $query->select($db->qn('xref.comment_id'));
        $query->from($db->qn('#__commentbox_comments_flags_xref', 'xref'));
        $query->join('INNER', $db->qn('#__commentbox_flags', 'flag') . ' ON ' . $db->qn('xref.flag_id') . ' = ' . $db->qn('flag.id'));
        $query->where($db->qn('xref.comment_id') . ' IN (' . implode(',', $commentIds) . ')');
        $db->setQuery($query);

        $xref = $db->loadObjectList();

        foreach ($comments as $comment) {
            $comment->flags = array_filter($xref, function ($flag) use ($comment) {
                return $flag->comment_id == $comment->id;
            });
        }
    }

    protected function addFlags($commentId, $flags)
    {
        $db = $this->getDatabase();

        foreach ($flags as $flag) {

            $query  = $db->getQuery(true);
            $query->select($db->qn('id'));
            $query->from($db->qn('#__commentbox_flags'));
            $query->where($db->qn('name').' = '.$db->q($flag->label));
            $db->setQuery($query);
            $flagId = $db->loadResult();

            if (!$flagId) {
                continue;
            }

            $query  = $db->getQuery(true);
            $query->insert($db->qn('#__commentbox_comments_flags_xref'));
            $query->columns($db->qn('comment_id').', '.$db->qn('flag_id').', '.$db->qn('score'));
            $query->values($db->q($commentId).', '.$db->q($flagId).', '.$db->q($flag->score));
            $db->setQuery($query);
            $db->execute();
        }
    }

    protected function removeFlags($commentId)
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->delete($db->qn('#__commentbox_comments_flags_xref'));
        $query->where($db->qn('comment_id').' = '.$db->q($commentId));
        $db->setQuery($query);
        $db->execute();
    }

}
