<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Component\CommentBox\Administrator\Helper;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Mail\MailTemplate;
use Joomla\CMS\Uri\Uri;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class NotificationHelper
{
    public static function comment($comment, $recipient)
    {
        $application = Factory::getApplication();

        $language = $application->getLanguage();
        $language->load('com_commentbox', JPATH_SITE. '/administrator/components/com_commentbox');

        $userName = $comment->created_by ? Factory::getUser($comment->created_by)->name : $comment->name;

        $model  = Factory::getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Page', 'Administrator', ['ignore_request' => true]);
        $page   = $model->getItem($comment->page_id);

        $data = [
            'PAGE_TITLE'     => $page->title,
            'RECIPIENT_NAME' => $recipient->name,
            'USER_NAME'      => $userName,
            'COMMENT_LINK'   => Uri::root().$page->link.'#comment-'.$comment->id,
            'SITE_NAME'      => $application->get('sitename'),
        ];

        $mailer = new MailTemplate('com_commentbox.comment.add', $language->getTag());
        $mailer->addTemplateData($data);
        $mailer->addRecipient($recipient->email, $recipient->name);

        try {
            $mailer->send();
        } catch (\Exception $exception) {
            Log::add(Text::_($exception->getMessage()), Log::WARNING, 'com_commentbox');
        }
    }

    public static function published($comment)
    {
        $application = Factory::getApplication();

        $language = $application->getLanguage();
        $language->load('com_commentbox', JPATH_SITE. '/administrator/components/com_commentbox');

        if ($comment->created_by) {
            $recipient = Factory::getUser($comment->created_by);
        } else {
            $recipient = (object) ['name' => $comment->name, 'email' => $comment->email];
        }

        $model  = Factory::getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Page', 'Administrator', ['ignore_request' => true]);
        $page   = $model->getItem($comment->page_id);

        $data = [
            'PAGE_TITLE'     => $page->title,
            'RECIPIENT_NAME' => $recipient->name,
            'COMMENT_LINK'   => Uri::root().$page->link.'#comment-'.$comment->id,
            'SITE_NAME'      => $application->get('sitename'),
        ];

        $mailer = new MailTemplate('com_commentbox.comment.publish', $language->getTag());
        $mailer->addTemplateData($data);
        $mailer->addRecipient($recipient->email, $recipient->name);

        try {
            $mailer->send();
        } catch (\Exception $exception) {
            Log::add(Text::_($exception->getMessage()), Log::WARNING, 'com_commentbox');
        }
    }

    public static function reply($reply)
    {
        $application = Factory::getApplication();

        $language = $application->getLanguage();
        $language->load('com_commentbox', JPATH_SITE. '/administrator/components/com_commentbox');

        $model  = Factory::getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Comment', 'Administrator', ['ignore_request' => true]);
        $parent = $model->getItem($reply->parent_id);

        $recipient = Factory::getUser($parent->created_by);

        if ($parent->created_by) {
            $recipient = Factory::getUser($parent->created_by);
        } else {
            $recipient = (object) ['name' => $parent->name, 'email' => $parent->email];
        }

        if ($reply->created_by) {
            $user = Factory::getUser($reply->created_by);
        } else {
            $user = (object) ['name' => $reply->name, 'email' => $reply->email];
        }

        if ($user->email == $recipient->email) {
            return;
        }

        $model  = Factory::getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Page', 'Administrator', ['ignore_request' => true]);
        $page   = $model->getItem($reply->page_id);

        $data = [
            'PAGE_TITLE'     => $page->title,
            'RECIPIENT_NAME' => $recipient->name,
            'USER_NAME'      => $user->name,
            'COMMENT_LINK'   => Uri::root().$page->link.'#comment-'.$reply->id,
            'SITE_NAME'      => $application->get('sitename'),
        ];

        $mailer = new MailTemplate('com_commentbox.comment.reply', $language->getTag());
        $mailer->addTemplateData($data);
        $mailer->addRecipient($recipient->email, $recipient->name);

        try {
            $mailer->send();
        } catch (\Exception $exception) {
            Log::add(Text::_($exception->getMessage()), Log::WARNING, 'com_commentbox');
        }
    }

    public static function flagged($comment)
    {
        $application = Factory::getApplication();

        $language = $application->getLanguage();
        $language->load('com_commentbox', JPATH_SITE. '/administrator/components/com_commentbox');

        if ($comment->created_by) {
            $recipient = Factory::getUser($comment->created_by);
        } else {
            $recipient = (object) ['name' => $comment->name, 'email' => $comment->email];
        }

        $model  = Factory::getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Page', 'Administrator', ['ignore_request' => true]);
        $page   = $model->getItem($comment->page_id);

        $data = [
            'PAGE_TITLE'     => $page->title,
            'RECIPIENT_NAME' => $recipient->name,
            'COMMENT_LINK'   => Uri::root().$page->link.'#comment-'.$comment->id,
            'SITE_NAME'      => $application->get('sitename'),
        ];

        $mailer = new MailTemplate('com_commentbox.comment.flagged', $language->getTag());
        $mailer->addTemplateData($data);
        $mailer->addRecipient($recipient->email, $recipient->name);

        try {
            $mailer->send();
        } catch (\Exception $exception) {
            Log::add(Text::_($exception->getMessage()), Log::WARNING, 'com_commentbox');
        }
    }

    public static function mention($comment)
    {
        $application = Factory::getApplication();

        $language = $application->getLanguage();
        $language->load('com_commentbox', JPATH_SITE. '/administrator/components/com_commentbox');

        if ($comment->created_by) {
            $user = Factory::getUser($comment->created_by);
        } else {
            $user = (object) ['name' => $comment->name, 'email' => $comment->email];
        }

        $model  = Factory::getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Page', 'Administrator', ['ignore_request' => true]);
        $page   = $model->getItem($comment->page_id);

        $dom  = new \DOMDocument('1.0', 'UTF-8');
        $html = '<!DOCTYPE html><html><head><meta http-equiv="Content-Type" content="charset=utf-8" /></head><body>' . $comment->text . '</body></html>';

        if (!$dom->loadHTML($html)) {
            return;
        }

        $spans = $dom->getElementsByTagName('span');

        foreach ($spans as $span) {
            $type = $span->getAttribute('data-type');

            if ($type !== 'mention') {
                continue;
            }

            $userId = (int) $span->getAttribute('data-id');

            if (!$userId) {
                continue;
            }

            $recipient = Factory::getUser($userId);

            $data = [
                'PAGE_TITLE'     => $page->title,
                'RECIPIENT_NAME' => $recipient->name,
                'USER_NAME'      => $user->name,
                'COMMENT_LINK'   => Uri::root().$page->link.'#comment-'.$comment->id,
                'SITE_NAME'      => $application->get('sitename'),
            ];

            $mailer = new MailTemplate('com_commentbox.user.mention', $language->getTag());
            $mailer->addTemplateData($data);
            $mailer->addRecipient($recipient->email, $recipient->name);


            try {
                $mailer->send();
            } catch (\Exception $exception) {
                Log::add(Text::_($exception->getMessage()), Log::WARNING, 'com_commentbox');
            }

        }
    }
}
