<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Component\CommentBox\Administrator\Helper;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Uri\Uri;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

class ModerationHelper
{
    public const FLAG_HARASSMENT     = 'harassment';
    public const FLAG_THREATS        = 'threats';
    public const FLAG_HATE           = 'hate';
    public const FLAG_DISCRIMINATION = 'discrimination';
    public const FLAG_VIOLENCE       = 'violence';
    public const FLAG_SEXUAL         = 'sexual';
    public const FLAG_SELF_HARM      = 'self-harm';
    public const FLAG_PROFANITY      = 'profanity';
    public const FLAG_ILLICIT        = 'illicit';
    public const FLAG_UNSAFE         = 'unsafe';
    public const FLAG_SPAM           = 'spam';
    public const FLAG_SENSITIVE      = 'sensitive';
    public const FLAG_INCOHERENT     = 'incoherent';
    public const FLAG_UNSUBSTANTIAL  = 'unsubstantial';
    public const FLAG_TEXT_IN_IMAGE  = 'text-in-image';

    public static $flagged    = null;
    public static $flags      = [];

    public static function isEnabled($state = null)
    {
        $user   = Factory::getApplication()->getIdentity();
        $params = ComponentHelper::getParams('com_commentbox');

        if (!$params->get('ai_moderation', 0)) {
            return false;
        }

        if ($state === -1 && $params->get('ai_moderation_skip_spam', 1)) {
            return false;
        }

        if ($user->authorise('core.edit.state', 'com_commentbox') && $params->get('ai_moderation_skip_moderators', 0)) {
            return false;
        }

        return true;
    }

    public static function autoPublish()
    {
        $params = ComponentHelper::getParams('com_commentbox');

        return $params->get('ai_moderation_auto_publish', 1);
    }

    public static function unpublish()
    {
        $params = ComponentHelper::getParams('com_commentbox');

        return $params->get('flagged_comments_action', 'unpublish') === 'unpublish';
    }

    public static function trash()
    {
        $params = ComponentHelper::getParams('com_commentbox');

        return $params->get('flagged_comments_action', 'unpublish') === 'trash';
    }

    public static function reject()
    {
        $params = ComponentHelper::getParams('com_commentbox');

        return $params->get('flagged_comments_action', 'unpublish') === 'reject';
    }

    public static function moderate($comment)
    {
        $params = ComponentHelper::getParams('com_commentbox');

        $service = $params->get('ai_moderation_service');

        if (!$service) {
            return;
        }

        if (\is_array($comment)) {
            $comment = (object) $comment;
        }

        $images  = self::getImages($comment->text);
        $text    = self::getText($comment->text);

        if (!$text && !\count($images)) {
            return;
        }

        PluginHelper::importPlugin('commentbox', $service);

        try {
            [$flagged, $flags] = Factory::getApplication()->triggerEvent('onCommentBoxModerate', [$text, $images]);
        } catch (\Exception $exception) {
            Log::add(Text::_($exception->getMessage()), Log::WARNING, 'com_commentbox');
            return;
        }

        self::$flagged = $flagged;
        self::$flags   = self::convertServiceFlags($flags, $service);

        return self::$flagged;
    }

    private static function getImages($text)
    {
        $images  = [];

        $pattern = '/<img\s+[^>]*src=["\']([^"\']+)["\']/i';

        if (preg_match_all($pattern, $text, $matches)) {

            foreach ($matches[1] as $src) {

                $image = strtolower($src);

                if (strpos($image, 'http') !== 0 && strpos($image, 'https') !== 0) {
                    $image = Uri::root(false).ltrim($image, '/');
                }

                $images[] = $image;
            }
        }

        return $images;
    }

    private static function getText($text)
    {
        // Convert links to text
        $pattern = '/<a\s+[^>]*href=["\']([^"\']+)["\'][^>]*>(.*?)<\/a>/is';

        $text = preg_replace_callback(
            $pattern,
            function ($matches) {
                $href     = $matches[1];
                $linkText = strip_tags($matches[2]);
                return $linkText. ' (' . $href . ')';
            },
            $text
        );

        // Remove all HTML tags
        $text = preg_replace('#<[^>]+>#', ' ', $text);
        $text = strip_tags($text);
        $text = trim($text);

        return $text;
    }

    private static function convertServiceFlags(array $serviceFlags, string $service): array
    {
        PluginHelper::importPlugin('commentbox', $service);

        try {
            $mapping = Factory::getApplication()->triggerEvent('onCommentBoxGetFlagsMapping');
        } catch (\Exception $exception) {
            Log::add(Text::_($exception->getMessage()), Log::WARNING, 'com_commentbox');
            return [];
        }

        $internalFlags = [];

        foreach ($serviceFlags as $serviceFlag) {

            if (isset($mapping[$serviceFlag->label])) {

                $mapped = $mapping[$serviceFlag->label];

                if (\is_array($mapped)) {

                    foreach ($mapped as $flag) {
                        $internalFlags[$flag] = (object) [
                            'label' => $flag,
                            'score' => $serviceFlag->score,
                        ];
                    }

                } else {
                    $internalFlags[$mapped] = (object) [
                        'label' => $mapped,
                        'score' => $serviceFlag->score,
                    ];
                }

            }
        }

        return array_values($internalFlags);
    }

}
