<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Component\CommentBox\Administrator\Helper;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\FileLayout;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class CommentBoxHelper
{
    public static function render()
    {
        if (!CommentBoxHelper::shouldRender()) {
            return '';
        }

        $page = CommentBoxHelper::getPage();

        if ($page && isset($page->state) && $page->state == 0) {
            return '';
        }

        CommentBoxHelper::loadLanguage();
        $options = CommentBoxHelper::getOptions($page);
        CommentBoxHelper::loadScripts($options);

        $styles = CommentBoxHelper::getEditorStyles($options);

        $commentboxBefore = CommentBoxHelper::loadModulePosition('commentbox-before');
        $commentboxAfter  = CommentBoxHelper::loadModulePosition('commentbox-after');

        $layout = new FileLayout('commentbox.application', null, ['component' => 'com_commentbox']);

        $application = Factory::getApplication();
        if ($application->isClient('site')) {
            $layout->addIncludePath(JPATH_SITE . '/templates/' . $application->getTemplate() . '/html/layouts');
        } else {
            $layout->addIncludePath(JPATH_SITE . '/components/com_commentbox/layouts');
        }

        return $layout->render(['commentboxBefore' => $commentboxBefore, 'commentboxAfter' => $commentboxAfter, 'styles' => $styles]);
    }

    private static function loadLanguage()
    {
        $language = Factory::getLanguage();
        $language->load('com_commentbox', JPATH_SITE.'/components/com_commentbox');

        Text::script('COM_COMMENTBOX_CANCEL');
        Text::script('COM_COMMENTBOX_SEND');
        Text::script('COM_COMMENTBOX_LEAVE_YOUR_COMMENT');
        Text::script('COM_COMMENTBOX_COMMENT_CANNOT_BE_EMPTY');
        Text::script('COM_COMMENTBOX_VIEW_MORE');
        Text::script('COM_COMMENTBOX_LOGIN_TO_COMMENT');
        Text::script('COM_COMMENTBOX_YOU_ARE_NOT_ALLOWED_TO_COMMENT');
        Text::script('COM_COMMENTBOX_PUBLISHED');
        Text::script('COM_COMMENTBOX_UNPUBLISHED');
        Text::script('COM_COMMENTBOX_REPLY');
        Text::script('COM_COMMENTBOX_EDIT');
        Text::script('COM_COMMENTBOX_UNPUBLISH');
        Text::script('COM_COMMENTBOX_PUBLISH');
        Text::script('COM_COMMENTBOX_SPAM');
        Text::script('COM_COMMENTBOX_NAME');
        Text::script('COM_COMMENTBOX_PLEASE_ENTER_YOUR_NAME');
        Text::script('COM_COMMENTBOX_EMAIL');
        Text::script('COM_COMMENTBOX_PLEASE_ENTER_YOUR_EMAIL');
        Text::script('COM_COMMENTBOX_INVALID_EMAIL');
        Text::script('COM_COMMENTBOX_REPLY_TO');
        Text::script('COM_COMMENTBOX_DELETE');
        Text::script('COM_COMMENTBOX_DELETE_CONFIRM');
        Text::script('COM_COMMENTBOX_REMOVE');
        Text::script('COM_COMMENTBOX_REMOVE_CONFIRM');
        Text::script('COM_COMMENTBOX_UPDATE');
        Text::script('COM_COMMENTBOX_LINK');

        Text::script('COM_COMMENTBOX_EDITOR_IMAGE');
        Text::script('COM_COMMENTBOX_EDITOR_BOLD');
        Text::script('COM_COMMENTBOX_EDITOR_ITALIC');
        Text::script('COM_COMMENTBOX_EDITOR_UNDERLINE');
        Text::script('COM_COMMENTBOX_EDITOR_STRIKETHROUGH');
        Text::script('COM_COMMENTBOX_EDITOR_CODE');
        Text::script('COM_COMMENTBOX_EDITOR_LIST');
        Text::script('COM_COMMENTBOX_EDITOR_ORDERED_LIST');
        Text::script('COM_COMMENTBOX_EDITOR_QUOTE');
        Text::script('COM_COMMENTBOX_TERMS_LABEL');
        Text::script('COM_COMMENTBOX_PLEASE_ACCEPT_THE_TERMS');

        Text::script('COM_COMMENTBOX_EDITOR_EMOJI');
        Text::script('COM_COMMENTBOX_EDITOR_GIF');
        Text::script('COM_COMMENTBOX_SEARCH_TENOR');
        Text::script('COM_COMMENTBOX_SEARCH_EMOJIS');
        Text::script('COM_COMMENTBOX_COMMENTS_ARE_CLOSED');

        if (CaptchaHelper::isEnabled()) {
            Text::script('COM_COMMENTBOX_ALTCHA_ARIA_LINK_LABEL');
            Text::script('COM_COMMENTBOX_ALTCHA_ERROR');
            Text::script('COM_COMMENTBOX_ALTCHA_EXPIRED');
            Text::script('COM_COMMENTBOX_ALTCHA_FOOTER');
            Text::script('COM_COMMENTBOX_ALTCHA_LABEL');
            Text::script('COM_COMMENTBOX_ALTCHA_VERIFIED');
            Text::script('COM_COMMENTBOX_ALTCHA_VERIFYING');
            Text::script('COM_COMMENTBOX_ALTCHA_WAIT_ALERT');
            Text::script('COM_COMMENTBOX_RECAPTCHA_CREDITS');
        }

        Text::script('COM_COMMENTBOX_LIKE');
        Text::script('COM_COMMENTBOX_DISLIKE');
        Text::script('COM_COMMENTBOX_LOVE');
        Text::script('COM_COMMENTBOX_FUNNY');
        Text::script('COM_COMMENTBOX_WOW');
        Text::script('COM_COMMENTBOX_SAD');
        Text::script('COM_COMMENTBOX_ANGRY');

        Text::script('COM_COMMENTBOX_UPVOTE');
        Text::script('COM_COMMENTBOX_DOWNVOTE');
        Text::script('COM_COMMENTBOX_SCORE');
    }

    private static function getOptions($page)
    {
        $application = Factory::getApplication();

        $params = ComponentHelper::getParams('com_commentbox');

        $item    = CommentBoxHelper::getItemData();
        $hash    = HashHelper::generateHash($item);
        $style   = CommentBoxHelper::getStyleData();
        $captcha = CommentBoxHelper::getCaptchaData();
        $tags    = CommentBoxHelper::getAllowedTags();
        $terms   = CommentBoxHelper::getTermsData();

        $options = [
            'item'           => $item,
            'hash'           => $hash,
            'style'          => $style,
            'direction'      => Factory::getDocument()->getDirection(),
            'client'         => $application->getName(),
            'comments'       => $application->isClient('administrator') ? CommentBoxHelper::getDemoComments() : null,
            'user'           => $application->isClient('administrator') ? CommentBoxHelper::getDemoUser() : null,
            'captcha'        => $captcha,
            'tags'           => $tags,
            'mentions'       => (bool) $params->get('mentions', 1),
            'emojis'         => (bool) $params->get('emojis', 1),
            'gifs'           => (bool) $params->get('gifs', 1) && $params->get('tenor_api_key'),
            'uploads'        => (bool) $params->get('image_uploads', 1),
            'moderation'     => (bool) $params->get('frontend_moderation', 1),
            'votes'          => (bool) $params->get('votes', 1),
            'votes_counters' => (string) $params->get('votes_counters', 'score'),
            'reactions'      => $params->get('reactions', 1) ? $params->get('enabled_reactions') : [],
            'login'          => Route::_('index.php?option=com_users&view=login&return=' . base64_encode(Uri::getInstance()->toString())),
            'terms'          => $terms,
            'closed'         => $page && $page->state == -1,
        ];

        return $options;
    }

    private static function getItemData()
    {
        $application = Factory::getApplication();
        $document    = Factory::getDocument();

        $title               = $document->getTitle();
        $siteNameInPageTitle = $application->get('sitename_pagetitles');

        if ($siteNameInPageTitle) {
            $siteName = $application->get('sitename');
            $search   = $siteNameInPageTitle == 1 ? Text::sprintf('JPAGETITLE', $siteName, '') : Text::sprintf('JPAGETITLE', '', $siteName);
            $title    = str_replace($search, '', $title);
        }

        $link      = substr(Uri::current(), \strlen(Uri::root()));
        $option    = $application->input->getCmd('option');
        $component = substr($option, 4);
        $view      = $application->input->getCmd('view');
        $key       = $application->input->getInt(CommentBoxHelper::getKeyName($option));

        return  [
            'title'     => $title,
            'link'      => $link,
            'component' => $component,
            'view'      => $view,
            'key'       => $key,
        ];
    }


    private static function getStyleData()
    {
        $params = ComponentHelper::getParams('com_commentbox');

        return  [
            'darkMode'      => (bool) $params->get('dark_mode', false),
            'componentSize' => $params->get('component_size', 'middle'),
            'colorPrimary'  => $params->get('color_primary', '#1677ff'),
            'borderRadius'  => (int) $params->get('border_radius', 6),
            'fontFamily'    => $params->get('font_family', "-apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, 'Noto Sans', sans-serif, 'Apple Color Emoji', 'Segoe UI Emoji', 'Segoe UI Symbol', 'Noto Color Emoji'"),
            'fontSize'      => (int) $params->get('font_size', 14),
        ];
    }

    private static function getCaptchaData()
    {
        $params  = ComponentHelper::getParams('com_commentbox');
        $options = ['enabled' => CaptchaHelper::isEnabled()];
        if ($options['enabled']) {
            $options['service'] = $params->get('captcha_service');
            if ($options['service'] === 'recaptcha') {
                $options['siteKey'] = $params->get('recaptcha_site_key', '');
            }
        }

        return $options;
    }

    private static function getTermsData()
    {
        $params  = ComponentHelper::getParams('com_commentbox');
        $user    = Factory::getApplication()->getIdentity();

        return $params->get('terms', 0) == 2 || ($params->get('terms', 0) == 1 && $user->guest);
    }
    private static function loadModulePosition($position)
    {
        $buffer = '';

        $modules = ModuleHelper::getModules($position);

        foreach ($modules as $module) {
            $buffer .= ModuleHelper::renderModule($module, ['style' => '']);
        }

        return $buffer;
    }

    private static function loadScripts($options)
    {
        HTMLHelper::_('form.csrf');

        $document = Factory::getDocument();
        $document->addScriptOptions('commentbox', $options);

        $wa = $document->getWebAssetManager();
        if (CaptchaHelper::isEnabled()) {
            CaptchaHelper::load();
        }

        //$wa->registerAndUseScript('commentbox', 'http://localhost:5173/src/main.js', [], ['type' => 'module']);
        $wa->registerAndUseScript('commentbox', 'commentbox/commentbox.min.js', [], ['type' => 'module']);
        $wa->registerAndUseScript('commentbox.links', 'commentbox/links.js', [], ['defer' => true]);
        $wa->registerAndUseScript('commentbox.counters', 'commentbox/counters.js', [], ['defer' => true]);

    }

    public static function counter($url)
    {
        $document = Factory::getDocument();
        $wa       = $document->getWebAssetManager();
        $wa->registerAndUseScript('commentbox.counters', 'commentbox/counters.js', [], ['defer' => true]);

        $language = Factory::getLanguage();
        $language->load('com_commentbox', JPATH_SITE.'/components/com_commentbox');

        Text::script('COM_COMMENTBOX_NO_COMMENTS');
        Text::script('COM_COMMENTBOX_ONE_COMMENT');
        Text::script('COM_COMMENTBOX_MULTIPLE_COMMENTS');

        $layout = new FileLayout('commentbox.counter', null, ['component' => 'com_commentbox']);
        $layout->addIncludePath(JPATH_SITE . '/templates/' . Factory::getApplication()->getTemplate() . '/html/layouts');

        return $layout->render(['url' => $url]);
    }

    private static function getDemoComments()
    {
        return [
            (object) [
                'id'        => 1,
                "parent_id" => 0,
                "text"      => "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus ligula nulla, molestie quis eros eu, ultrices suscipit neque. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia curae; Praesent tempor convallis urna in aliquet.",
                "state"     => 1,
                "name"      => "John Smith",
                "created"   => "2024-08-11 15:10:53",
                "level"     => 1,
                "upvotes"   => 5,
                "downvotes" => 2,
                "score"     => 3,
                "reactions" => [],
                "replies"   => [
                    (object) [
                        'id'           => 3,
                        "parent_id"    => 0,
                        "text"         => "Curabitur tincidunt in tortor at aliquet. Sed iaculis lacinia euismod.",
                        "state"        => 1,
                        "name"         => "Jane Doe",
                        "created"      => "2024-08-11 15:10:53",
                        "level"        => 1,
                        "upvotes"      => 12,
                        "downvotes"    => 0,
                        "score"        => 12,
                        "reactions"    => ['love' => 4, 'funny' => 2],
                        "liked"        => true,
                        "replies"      => [],
                        "mine"         => true,
                        "relativeDate" => "3 weeks ago.",
                        "avatar"       => UserHelper::getAvatar('info@firecoders.com'),

                    ],
                ],
                "mine"         => false,
                "relativeDate" => "3 weeks ago.",
                "avatar"       => UserHelper::getAvatar('info@firecoders.com'),

            ],
            (object) [
                'id'           => 2,
                "parent_id"    => 0,
                "text"         => "Pellentesque scelerisque ultrices sapien in semper. Sed ac elit ut sem feugiat mattis a non quam. Ut tortor libero, accumsan sed risus vitae, aliquam sodales nulla",
                "state"        => 1,
                "name"         => "Jane Doe",
                "created"      => "2024-08-11 15:10:53",
                "level"        => 1,
                "upvotes"      => 1,
                "downvotes"    => 0,
                "score"        => 1,
                "reactions"    => ['love' => 1],
                "upvoted"      => true,
                "replies"      => [],
                "mine"         => false,
                "relativeDate" => "3 weeks ago.",
                "avatar"       => UserHelper::getAvatar('info@firecoders.com'),

            ],
        ];
    }

    private static function getDemoUser()
    {
        return (object)[
            'name'        => 'Jane Doe',
            'guest'       => false,
            'avatar'      => UserHelper::getAvatar('info@firecoders.com'),
            'permissions' => ['add' => true],
        ];
    }

    public static function getKeyName($option)
    {
        if ($option === 'com_virtuemart') {
            return 'virtuemart_product_id';
        } elseif ($option === 'com_hikashop') {
            return 'cid';
        }

        return 'id';
    }

    public static function getAllowedTags()
    {
        $allowed = [];

        $tags = ['p', 'a', 'img', 'strong', 'em', 's', 'u', 'code', 'ul', 'ol', 'blockquote'];

        foreach ($tags as $tag) {

            $text = '<'.$tag.'></'.$tag.'>';

            if (ComponentHelper::filterText($text) === $text) {
                $allowed[] = $tag;
            }
        }

        return $allowed;
    }

    public static function copyrights()
    {
        $db    = Factory::getDbo();
        $query = $db->getQuery(true);
        $query->select($db->qn('manifest_cache'))->from($db->qn('#__extensions'))->where($db->qn('element') . ' = ' . $db->q('pkg_commentbox'));
        $db->setQuery($query);
        $manifest = json_decode($db->loadResult());

        $date     = Factory::getDate();
        $html     = '<div class="text-center help-block"><a target="_blank" href="https://www.firecoders.com/joomla-extensions/commentbox">' . $manifest->name . ' v' . $manifest->version . '</a> | Copyright &copy; 2016 - ' . $date->format('Y') . ' <a target="_blank" href="https://www.firecoders.com">Firecoders</a></div>';
        $html .= '<div class="text-center help-block">If you like CommentBox, please post a review at the <a href="https://extensions.joomla.org/extension/commentbox/" target="_blank">Joomla Extensions Directory</a>.</div>';

        return $html;
    }

    private static function getEditorStyles($options)
    {
        $style = '
            .commentbox-comment-content-detail img {
                max-width: 100%;
                max-height: 300px;
            }

            .tiptap {
                outline: none;
            }

            .tiptap > p {
                margin: 0 0 12px 0 !important;
            }

            /* Basic editor styles */
            .tiptap:first-child {
                margin-top: 0;
            }

            /* List styles */
            .tiptap ul,
            .tiptap ol {
                padding: 0 1rem;
                margin: 1.25rem 1rem 1.25rem 0.4rem;
            }

            .tiptap li p {
                margin-top: 0.25em;
                margin-bottom: 0.25em;
            }

            /* Code and preformatted text styles */
            .tiptap code {
                background-color: rgba(88, 5, 255, 0.05);
                border-radius: .4rem;
                color: #000;
                font-size: .85rem;
                padding: .25em .3em;
            }

            .tiptap pre {
                background: #000;
                border-radius: .5rem;
                color: #fff;
                font-family: \'JetBrainsMono\', monospace;
                margin: 1.5rem 0;
                padding: .75rem 1rem;
            }

            .tiptap pre code {
                background: none;
                color: inherit;
                font-size: 0.8rem;
                padding: 0;
            }

            .tiptap blockquote {
                border-left: 3px solid rgba(61, 37, 20, .12);
                margin: 1.5rem 0;
                padding-left: 1rem;
            }

            .tiptap hr {
                border: none;
                border-top: 1px solid rgba(61, 37, 20, .12);
                margin: 2rem 0;
            }

            /* Placeholder (at the top) */
            .tiptap p.is-editor-empty:first-child::before {
                color: rgba(0, 0, 0, 0.25);
                content: attr(data-placeholder);
                float: left;
                height: 0;
                pointer-events: none;
            }

            .tiptap img {
                display: block;
                max-width: 100%;
                max-height: 200px;
            }
        ';

        if ($options['style']['darkMode']) {
            $style .= '
                @media (prefers-color-scheme: dark) {.tiptap p.is-editor-empty:first-child::before {color: rgba(255, 255, 255, 0.25);}}
            ';
        }

        return $style;
    }

    private static function shouldRender()
    {
        $application = Factory::getApplication();

        if ($application->isClient('administrator') && $application->input->getCmd('option') === 'com_config') {
            return true;
        }

        $params = ComponentHelper::getParams('com_commentbox');

        $key = $application->input->getCmd('option').'_'.$application->input->getCmd('view');

        $default = $key === 'com_content_article' ? 1 : 0;

        if (!$params->get($key, $default)) {
            return false;
        }

        $Itemid = $application->input->getInt('Itemid');

        if ($Itemid && \in_array($Itemid, $params->get('exclusions', []))) {
            return false;
        }

        return true;
    }

    private static function getPage()
    {
        $application = Factory::getApplication();

        $option    = $application->input->getCmd('option');
        $component = substr($option, 4);
        $view      = $application->input->getCmd('view');
        $key       = $application->input->getInt(CommentBoxHelper::getKeyName($option));

        $model = $application->bootComponent('com_commentbox')->getMVCFactory()->createModel('Page', 'Administrator', ['ignore_request' => true]);
        $page  = $model->getItemByResource($component, $view, $key);

        return $page;
    }
}
