<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Plugin\ActionLog\CommentBox\Extension;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\UserFactoryAwareTrait;
use Joomla\Component\Actionlogs\Administrator\Plugin\ActionLogPlugin;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Utilities\ArrayHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


final class CommentBox extends ActionLogPlugin implements SubscriberInterface
{
    use DatabaseAwareTrait;
    use UserFactoryAwareTrait;

    public static function getSubscribedEvents(): array
    {
        if (!ComponentHelper::isEnabled('com_commentbox')) {
            return [];
        }

        return [
            'onContentAfterSave'             => 'onContentAfterSave',
            'onContentChangeState'           => 'onContentChangeState',
            'onContentAfterDelete'           => 'onContentAfterDelete',
            'onCommentBoxPageDeleteComments' => 'onPageDeleteComments',
        ];
    }

    public function onContentAfterSave(Event $event): void
    {
        [$context, $item, $isNew] = array_values($event->getArguments());

        if ($context === 'com_commentbox.comment') {
            $this->onCommentAfterSave($context, $item, $isNew);
        } elseif ($context === 'com_commentbox.page') {
            $this->onPageAfterSave($context, $item, $isNew);
        }
    }

    public function onContentChangeState(Event $event): void
    {
        [$context, $pks, $value] = array_values($event->getArguments());

        if ($context === 'com_commentbox.comment') {
            $this->onCommentChangeState($context, $pks, $value);
        } elseif ($context === 'com_commentbox.page') {
            $this->onPageChangeState($context, $pks, $value);
        }
    }

    public function onContentAfterDelete(Event $event): void
    {
        [$context, $item] = array_values($event->getArguments());

        if ($context === 'com_commentbox.comment') {
            $this->onCommentAfterDelete($context, $item);
        } elseif ($context === 'com_commentbox.page') {
            $this->onPageAfterDelete($context, $item);
        }
    }

    private function onCommentAfterSave($context, $comment, $isNew): void
    {
        $data  = $this->app->input->get('data', json_decode($this->app->input->json->getRaw(), true), 'array');

        $message = [
            'action'      => $isNew ? 'add' : 'update',
            'pageTitle'   => $data['item']['title'],
            'pageLink'    => 'index.php?option=com_commentbox&task=page.edit&id='.$comment->page_id,
            'commentId'   => $comment->id,
            'commentLink' => Uri::root().$data['item']['link']. '#comment-'.$comment->id,
        ];

        $messageLanguageKey = $isNew ? 'PLG_ACTIONLOG_COMMENTBOX_COMMENT_ADDED' : 'PLG_ACTIONLOG_COMMENTBOX_COMMENT_UPDATED';

        $this->addLog([$message], $messageLanguageKey, $context);
    }

    private function onCommentChangeState($context, $pks, $value): void
    {
        $action = '';

        switch ($value) {
            case 1:
                $messageLanguageKey = 'PLG_ACTIONLOG_COMMENTBOX_COMMENT_PUBLISHED';
                $action             = 'publish';
                break;
            case 0:
                $messageLanguageKey = 'PLG_ACTIONLOG_COMMENTBOX_COMMENT_UNPUBLISHED';
                $action             = 'unpublish';
                break;
            case -1:
                $messageLanguageKey = 'PLG_ACTIONLOG_COMMENTBOX_COMMENT_SPAM';
                $action             = 'spam';
                break;
            case -2:
                $messageLanguageKey = 'PLG_ACTIONLOG_COMMENTBOX_COMMENT_TRASHED';
                $action             = 'trash';
                break;
        }

        if (!$action) {
            return;
        }

        $model = $this->getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Comments', 'Administrator', ['ignore_request' => true]);
        $model->setState('filter.id', ArrayHelper::toInteger($pks));
        $items = $model->getItems();

        $comments = [];
        foreach ($items as $item) {
            $comments[$item->id] = $item;
        }

        $messages = [];

        foreach ($pks as $pk) {
            $message = [
                'action'      => $action,
                'commentId'   => $pk,
                'commentLink' => $comments[$pk]->pageLink. '#comment-'.$pk,
                'pageTitle'   => $comments[$pk]->pageTitle,
                'pageLink'    => 'index.php?option=com_commentbox&task=page.edit&id='.$comments[$pk]->page_id,
            ];

            $messages[] = $message;
        }

        $this->addLog($messages, $messageLanguageKey, $context);
    }

    private function onCommentAfterDelete($context, $comment): void
    {
        $pageTitle = '';
        $pageLink  = '';

        $model = $this->getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Page', 'Administrator', ['ignore_request' => true]);
        $item  = $model->getItem($comment->page_id);

        if ($item) {
            $pageTitle = $item->title;
            $pageLink  = 'index.php?option=com_commentbox&task=page.edit&id='.$item->id;
        }

        $message = [
            'action'    => 'delete',
            'commentId' => $comment->id,
            'pageTitle' => $pageTitle,
            'pageLink'  => $pageLink,
        ];

        $this->addLog([$message], 'PLG_ACTIONLOG_COMMENTBOX_COMMENT_DELETED', $context);
    }

    private function onPageAfterSave($context, $page, $isNew): void
    {
        if ($this->app->isClient('site')) {
            return;
        }

        $data  = $this->app->input->get('data', json_decode($this->app->input->json->getRaw(), true), 'array');

        $message = [
            'action'    => $isNew ? 'add' : 'update',
            'pageTitle' => $page->title,
            'pageLink'  => 'index.php?option=com_commentbox&task=page.edit&id='.$page->id,
        ];

        $messageLanguageKey = $isNew ? 'PLG_ACTIONLOG_COMMENTBOX_PAGE_ADDED' : 'PLG_ACTIONLOG_COMMENTBOX_PAGE_UPDATED';

        $this->addLog([$message], $messageLanguageKey, $context);
    }

    private function onPageChangeState($context, $pks, $value): void
    {
        $action = '';

        switch ($value) {
            case 1:
                $messageLanguageKey = 'PLG_ACTIONLOG_COMMENTBOX_PAGE_COMMENTS_ENABLED';
                $action             = 'enable';
                break;
            case 0:
                $messageLanguageKey = 'PLG_ACTIONLOG_COMMENTBOX_PAGE_COMMENTS_DISABLED';
                $action             = 'disable';
                break;
            case -1:
                $messageLanguageKey = 'PLG_ACTIONLOG_COMMENTBOX_PAGE_COMMENTS_CLOSED';
                $action             = 'close';
                break;
        }

        if (!$action) {
            return;
        }

        $model = $this->getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Pages', 'Administrator', ['ignore_request' => true]);
        $model->setState('filter.id', ArrayHelper::toInteger($pks));
        $items = $model->getItems();

        $pages = [];
        foreach ($items as $item) {
            $pages[$item->id] = $item;
        }

        $messages = [];

        foreach ($pks as $pk) {
            $message = [
                'action'    => $action,
                'pageTitle' => $pages[$pk]->title,
                'pageLink'  => 'index.php?option=com_commentbox&task=page.edit&id='.$pk,
            ];

            $messages[] = $message;
        }

        $this->addLog($messages, $messageLanguageKey, $context);
    }

    private function onPageAfterDelete($context, $page): void
    {
        $message = [
            'action'    => 'delete',
            'pageTitle' => $page->title,
        ];

        $this->addLog([$message], 'PLG_ACTIONLOG_COMMENTBOX_PAGE_DELETED', $context);
    }

    public function onPageDeleteComments(Event $event)
    {
        [$context, $pageIds] = array_values($event->getArguments());

        $model = $this->getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Page', 'Administrator', ['ignore_request' => true]);

        foreach ($pageIds as $pageId) {
            $page  = $model->getItem($pageId);

            $message = [
                'action'    => 'deleteComments',
                'pageTitle' => $page->title,
                'pageLink'  => 'index.php?option=com_commentbox&task=page.edit&id='.$page->id,
            ];

            $messages[] = $message;
        }

        $this->addLog($messages, 'PLG_ACTIONLOG_COMMENTBOX_PAGE_COMMENTS_DELETED', $context);
    }

}
