<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Plugin\CommentBox\Altcha\Extension;

\defined('_JEXEC') or die;

use Firecoders\Plugin\CommentBox\Altcha\Library\Altcha as AltchaService;
use Firecoders\Plugin\CommentBox\Altcha\Library\ChallengeOptions;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;

final class Altcha extends CMSPlugin implements SubscriberInterface
{
    public static function getSubscribedEvents(): array
    {
        return [
            'onCommentBoxCaptchaChallenge' => 'onCaptchaChallenge',
            'onCommentBoxCaptchaVerify'    => 'onCaptchaVerify',
            'onCommentBoxCaptchaLoad'      => 'onCaptchaLoad',
            'onCommentBoxSpamCheck'        => 'onSpamCheck',
        ];
    }

    public function onCaptchaChallenge(Event $event): void
    {
        $application = $this->getApplication();

        $options = new ChallengeOptions([
            'hmacKey'   => $application->get('secret'),
            'maxNumber' => 50000,
        ]);

        try {
            $challenge = AltchaService::createChallenge($options);
        } catch (Exception $e) {
            $challenge = null;
        }

        $event->setArgument('result', $challenge);
    }

    public function onCaptchaVerify(Event $event): void
    {
        $application = $this->getApplication();

        [$solution] = $event->getArguments();

        try {
            $payload  = json_decode(base64_decode($solution), true);
            $verified = AltchaService::verifySolution($payload, $application->get('secret'), true);
        } catch (Exception $e) {
            $verified = false;
        }

        $event->setArgument('result', $verified);
    }

    public function onCaptchaLoad(): void
    {
        $document = Factory::getDocument();

        $wa = $document->getWebAssetManager();
        $wa->registerAndUseScript('altcha', 'commentbox/altcha.min.js', [], ['type' => 'module']);
    }

    public function onSpamCheck(Event $event): void
    {
        [$comment] = $event->getArguments();

        $params = ComponentHelper::getParams('com_commentbox');
        $apiKey = $params->get('altcha_api_key');

        if (!$apiKey) {
            throw new \Exception(Text::_('COM_COMMENTBOX_ERROR_ANTI_SPAM_SERVICE_API_KEY_NOT_SET'), 500);
        }

        $parts = explode('@', $comment->email);
        $email = '@'.$parts[1];

        $data = [
            'text'         => $comment->text,
            'email'        => $email,
            'ipAddress'    => $comment->ip,
            'disableRules' => ['text.HTML'],
        ];

        $http = HttpFactory::getHttp();

        $response = $http->post('https://eu.altcha.org/api/v1/classify', json_encode($data), ['Content-Type' => 'application/json', 'Authorization' => 'Bearer '.$apiKey, 'Referer' => Uri::root(false)], 5);

        if ($response->code < 200 || $response->code >= 300) {
            throw new \Exception(Text::_('COM_COMMENTBOX_ERROR_COMMUNICATING_WITH_ANTI_SPAM_SERVICE'), 500);
        }

        $body   = json_decode($response->body);
        $result = isset($body->score) && $body->score > 2 ? true : false;

        $event->setArgument('result', $result);
    }
}
