<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Plugin\Content\CommentBox\Extension;

use Firecoders\Component\CommentBox\Administrator\Helper\CommentBoxHelper;
use Firecoders\Component\CommentBox\Administrator\Helper\NotificationHelper;
use Firecoders\Component\CommentBox\Administrator\Helper\UrlHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Router\Route;
use Joomla\CMS\User\CurrentUserTrait;
use Joomla\Component\Content\Site\Helper\RouteHelper;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Utilities\ArrayHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

final class CommentBox extends CMSPlugin implements SubscriberInterface
{
    use CurrentUserTrait;

    public static function getSubscribedEvents(): array
    {
        if (!ComponentHelper::isEnabled('com_commentbox')) {
            return [];
        }

        return [
            'onContentBeforeDisplay' => 'onContentBeforeDisplay',
            'onContentAfterSave'     => 'onContentAfterSave',
            'onContentChangeState'   => 'onContentChangeState',
        ];
    }

    public function onContentAfterSave(Event $event): void
    {
        [$context, $item, $isNew] = array_values($event->getArguments());

        if ($context !== 'com_commentbox.comment') {
            return;
        }

        if ($isNew) {

            $this->onCommentAdded($item);

            if ($item->state == 1) {
                $this->onCommentPublished($item);
            }
        }

        if ($item->flagged) {
            $this->onCommentFlagged($item);
        }
    }

    public function onContentChangeState(Event $event): void
    {
        [$context, $pks, $value] = array_values($event->getArguments());

        if ($context !== 'com_commentbox.comment') {
            return;
        }

        if ($value !== 1) {
            return;
        }

        $model = $this->getApplication()->bootComponent('commentbox')->getMVCFactory()->createModel('Comments', 'Administrator', ['ignore_request' => true]);
        $model->setState('filter.id', ArrayHelper::toInteger($pks));
        $items = $model->getItems();

        foreach ($items as $item) {
            $this->onCommentPublished($item);
        }

    }

    private function onCommentAdded($comment)
    {
        $params = ComponentHelper::getParams('com_commentbox');

        if (!$params->get('notify_new')) {
            return;
        }

        $userGroups = $params->get('notify_new_user_groups', []);

        if (!\count($userGroups)) {
            return;
        }

        $model = $this->getApplication()->bootComponent('users')->getMVCFactory()->createModel('Users', 'Administrator', ['ignore_request' => true]);
        $model->setState('filter.active', 0);
        $model->setState('filter.state', 0);
        $model->setState('filter.groups', $userGroups);
        $users = $model->getItems();

        foreach ($users as $user) {
            if ($user->id == $comment->created_by) {
                continue;
            }
            NotificationHelper::comment($comment, $user);
        }
    }

    private function onCommentPublished($comment)
    {
        $params = ComponentHelper::getParams('com_commentbox');

        $user = $this->getCurrentUser();

        if ($params->get('notify_publish') && $comment->created_by != $user->id) {
            NotificationHelper::published($comment);
        }

        if ($params->get('notify_reply') && $comment->parent_id) {
            NotificationHelper::reply($comment);
        }

        if ($params->get('notify_mention') && strpos($comment->text, 'data-type="mention"')) {
            NotificationHelper::mention($comment);
        }
    }

    private function onCommentFlagged($comment)
    {
        $params = ComponentHelper::getParams('com_commentbox');

        if (!$params->get('notify_flagged')) {
            return;
        }

        NotificationHelper::flagged($comment);
    }

    public function onContentBeforeDisplay(Event $event): void
    {
        $params = ComponentHelper::getParams('com_commentbox');

        if (!$params->get('com_content_article', 1) || !$params->get('com_content_article_counters', 1)) {
            return;
        }
        [$context, $item] = array_values($event->getArguments());

        $contexts = ['com_content.article', 'com_content.category', 'com_content.featured', 'com_content.archive'];

        if (!\in_array($context, $contexts)) {
            return;
        }

        if ((isset($item->access)) && !\in_array($item->access, $this->getCurrentUser()->getAuthorisedViewLevels())) {
            return;
        }

        $link = Route::_(RouteHelper::getArticleRoute($item->slug, $item->catid, $item->language));

        [$component, $view, $key, $Itemid] = UrlHelper::parse($link);

        if ($Itemid && \in_array($Itemid, $params->get('exclusions', []))) {
            return;
        }

        $model = $this->getApplication()->bootComponent('com_commentbox')->getMVCFactory()->createModel('Page', 'Administrator', ['ignore_request' => true]);
        $page  = $model->getItemByResource('content', 'article', $item->id);

        if ($page && isset($page->state) && $page->state == 0) {
            return;
        }

        $html = CommentBoxHelper::counter($link);

        $event->setArgument('result', array_merge($event->getArgument('result'), [$html]));
    }
}
