<?php

/**
 * @author      Lefteris Kavadas
 * @copyright   Copyright (c) 2016 - 2025 Lefteris Kavadas / firecoders.com
 * @license     GNU General Public License version 3 or later
 */

namespace Firecoders\Plugin\System\CommentBox\Extension;

use Firecoders\Component\CommentBox\Administrator\Helper\CommentBoxHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Extension\ExtensionHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

final class CommentBox extends CMSPlugin implements SubscriberInterface
{
    protected $autoloadLanguage = true;

    public static function getSubscribedEvents(): array
    {
        if (!ComponentHelper::isEnabled('com_commentbox')) {
            return [];
        }

        return [
            'onAfterInitialise'    => 'onAfterInitialise',
            'onBeforeRender'       => 'onBeforeRender',
            'onContentPrepareForm' => 'onContentPrepareForm',
        ];
    }

    public function onAfterInitialise()
    {
        $extension = ExtensionHelper::getExtensionRecord('pkg_commentbox', 'package');
        $manifest  = json_decode($extension->manifest_cache);

        \define('COMMENTBOX_VERSION', $manifest->name === 'CommentBox PRO' ? 'pro' : 'free');

        if (COMMENTBOX_VERSION === 'pro') {
            return;
        }

        $params = ComponentHelper::getParams('com_commentbox');
        $params->set('com_flexicontent_item', 0);
        $params->set('com_sppagebuilder_page', 0);
        $params->set('com_easyblog_entry', 0);
        $params->set('com_virtuemart_productdetails', 0);
        $params->set('com_eshop_product', 0);
        $params->set('com_hikashop_product', 0);
        $params->set('com_phocacart_item', 0);
        $params->set('com_joomcck_record', 0);
        $params->set('anti_spam', 0);
        $params->set('mentions', 0);
        $params->set('notify_mention', 0);
        $params->set('notify_flagged', 0);
        $params->set('gifs', 0);
        $params->set('image_uploads', 0);
        $params->set('frontend_moderation', 0);
        $params->set('ai_moderation', 0);
        $params->set('dark_mode', 0);
        $params->set('reactions', 0);
        $params->set('enabled_reactions', []);
    }

    public function onBeforeRender(Event $event)
    {
        $application = Factory::getApplication();

        if (!$application->isClient('site')) {
            return;
        }

        $document = Factory::getDocument();

        if ($document->getType() !== 'html') {
            return;
        }

        $headers = $application->getHeaders();
        foreach ($headers as $header) {
            if (strtolower($header['name']) == 'status' && $header['value'] == 403) {
                return;
            }
        }

        $buffer = $document->getBuffer('component');

        if (strpos($buffer, 'id="commentbox"') !== false) {
            return;
        }

        $commentbox = CommentBoxHelper::render();

        if ($commentbox) {
            $document->setBuffer($buffer.$commentbox, ['type' => 'component']);
        }

    }

    public function onContentPrepareForm(Event $event): void
    {
        if (!\defined('COMMENTBOX_VERSION')) {
            return;
        }

        if (COMMENTBOX_VERSION === 'pro') {
            return;
        }

        $application = Factory::getApplication();

        if (!$application->isClient('administrator')) {
            return;
        }

        $component = $application->input->getCmd('component');

        $form = $event->getArgument('form');

        if ($form->getName() !== 'com_config.component' || $component !== 'com_commentbox') {
            return;
        }

        $form->removeField('com_flexicontent_item');
        $form->removeField('com_sppagebuilder_page');
        $form->removeField('com_easyblog_entry');
        $form->removeField('com_virtuemart_productdetails');
        $form->removeField('com_eshop_product');
        $form->removeField('com_hikashop_product');
        $form->removeField('com_phocacart_item');
        $form->removeField('anti_spam');
        $form->removeField('mentions');
        $form->removeField('notify_mention');
        $form->removeField('notify_flagged');
        $form->removeField('gifs');
        $form->removeField('tenor_api_key');
        $form->removeField('image_uploads');
        $form->removeField('editor_permissions_note');
        $form->removeField('frontend_moderation');
        $form->removeField('ai_moderation');
        $form->removeField('dark_mode');
        $form->removeField('reactions');
        $form->removeField('enabled_reactions');

        $xml = new \SimpleXMLElement('<field name="integration_pro_note" label="COM_COMMENTBOX_NOTE" description="COM_COMMENTBOX_INTEGRATION_FEATURES_PRO" type="note" />');
        $form->setField($xml, null, false, 'integration');

        $xml = new \SimpleXMLElement('<field name="anti_spam_pro_note" label="COM_COMMENTBOX_NOTE" description="COM_COMMENTBOX_ANTISPAM_FEATURES_PRO" type="note" />');
        $form->setField($xml, null, false, 'spam');

        $xml = new \SimpleXMLElement('<field name="ai_moderation_pro_note" description="COM_COMMENTBOX_AI_MODERATION_FEATURES_PRO" type="note" />');
        $form->setField($xml, null, false, 'ai_moderation');

        $xml = new \SimpleXMLElement('<field name="editor_pro_note" label="COM_COMMENTBOX_NOTE" description="COM_COMMENTBOX_EDITOR_FEATURES_PRO" type="note" />');
        $form->setField($xml, null, false, 'editor');

        $xml = new \SimpleXMLElement('<field name="reactions_pro_note" label="COM_COMMENTBOX_NOTE" description="COM_COMMENTBOX_REACTIONS_FEATURES_PRO" type="note" />');
        $form->setField($xml, null, false, 'voting_and_reactions');

        $xml = new \SimpleXMLElement('<field name="style_pro_note" label="COM_COMMENTBOX_NOTE" description="COM_COMMENTBOX_STYLE_FEATURES_PRO" type="note" />');
        $form->setField($xml, null, false, 'style');

        $xml = new \SimpleXMLElement('<field name="other_pro_note" label="COM_COMMENTBOX_NOTE" description="COM_COMMENTBOX_OTHER_FEATURES_PRO" type="note" />');
        $form->setField($xml, null, false, 'other');
    }

}
